<?php

namespace Wi\Front\UserBundle\Controller;

use Wi\Front\CoreBundle\Core\Controller\FrontController;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\UserBundle\Entity\User;
use Wi\Admin\UserBundle\User\UserManager;
use Wi\Front\UserBundle\Form\RegisterType;
use Wi\Front\UserBundle\Service\Mailer;
use Wi\Front\UserBundle\Form\ForgotPasswordType;
use Wi\Front\UserBundle\Form\ResetPasswordType;

/**
 * Security Controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class SecurityController extends FrontController
{
    /**
     * Logowanie.
     *
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function loginAction(Request $request)
    {
        // Get authentication utils.
        $authenticationUtils = $this->get('security.authentication_utils');

        // Get the login error if there is one.
        $error = $authenticationUtils->getLastAuthenticationError();

        // Last username entered by the user.
        $lastUsername = $authenticationUtils->getLastUsername();

        return $this->render($this->getTemplate('Users/Security/login'), [
            'error' => $error,
            'lastUsername' => $lastUsername,
        ]);
    }

    /**
     * Rejestracja.
     *
     * @param Request $request
     * @param UserManager $userManager
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function registerAction(Request $request, UserManager $userManager)
    {
        // Create new object.
        $user = new User();

        // Generowanie formularza edycji zmiany hasła użytkownika.
        $form = $this->createForm(RegisterType::class, $user);
        $form->handleRequest($request);

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie regulaminu.
        $regulations = $em->getRepository('WiAdminPageBundle:Page')->findOneById(5);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Rejestracja użytkownika.
            $result = $userManager->register($user);

            // Ustawienie komunikatu o powodzeniu lub błędzie.
            if ($result) {
                $this->addFlash('register.success', 'Konto zostało utworzone.');
            } else {
                $this->addFlash('register.error', 'Wystąpił błąd.');
            }

            return $this->redirectToRoute('wi_front_user_register');
        }

        return $this->render($this->getTemplate('Users/Security/register'), [
            'form' => $form->createView(),
            'regulations' => $regulations,
        ]);
    }

    /**
     * Aktywacja konta użytkownika.
     *
     * @param Request $request
     * @param UserManager $userManager
     * @param User $user
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function activateAction(Request $request, UserManager $userManager, User $user)
    {
        // Aktywacja użytkownika.
        $userManager->activate($user);

        // Informacja o potwierdzeniu adresu e-mail.
        $this->addFlash('activate.success', 'Adres e-mail został zweryfikowany.');

        return $this->redirectToRoute('wi_front_user_login');
    }

    /**
     * Przywracanie hasła użytkownika.
     *
     * @param Request $request
     * @param Mailer $mailer
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function forgotPasswordAction(Request $request, Mailer $mailer)
    {
        // Utworzenie formularza.
        $form = $this->createForm(ForgotPasswordType::class);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Pobranie użytkownika na podstawie adresu e-mail.
            $user = $em->getRepository('WiAdminUserBundle:User')
                ->findOneByEmail($form->get('email')->getData())
            ;

            if (is_null($user)) { // Błąd, ponieważ nie istnieje użytkownik o podanym adresie e-mail.
                $form->addError(
                    new FormError('Użytkownik o podanym adresie nie istnieje')
                );
            } else { // Znaleziono pasującego użytkownika.
                // Ustawienie hasha.
                $user->setHashCode(hash('sha256', md5(
                    time()
                    . $this->get('session')->getId()
                    . $user->getEmail()
                    . time()
                    . rand()
                )));

                // Zapis.
                $em->persist($user);
                $em->flush();

                // Wysłanie wiadomości e-mail.
                $res = $mailer->sendForgotPasswordEmail($user);

                // Ustawienie komunikatu o powodzeniu lub błędzie.
                if ($res) {
                    $this->addFlash('user.forgot.success', 'Wiadomość została wysłana.');

                    // Zapisanie logu o wygenerowaniu linku do zmiany hasła.
                    $this->get('wi.front.user.logger')->create(16, [
                        'user_id' => $user->getId(),
                        'ip' => $request->server->get('REMOTE_ADDR'),
                        'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                        'descriptions' => 'Link for reset password has been generated.',
                    ]);
                } else {
                    $this->addFlash('user.forgot.error', 'Wystąpił błąd.');
                }
            }
        }

        return $this->render($this->getTemplate('Users/Security/forgot_password'), [
            'form' => $form->createView(),
        ]);
    }

    /**
     * Resetowanie hasła.
     *
     * @param Request $request
     * @param User $user Default 'null'.
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function resetPasswordAction(Request $request, User $user = null)
    {
        if (is_null($user)) {
            throw $this->createNotFoundException('Podana strona nie została odnaleziona');
        }

        // Utworzenie formularza.
        $form = $this->createForm(ResetPasswordType::class, $user);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted()) {
            if ($form->isValid()) {
                $user
                    ->setPassword($this->get('security.password_encoder')
                        ->encodePassword($user, $user->getPassword())
                    )
                    ->setHashCode(null)
                ;

                // Utworzenie instancji EntityManager-a.
                $em = $this->getDoctrine()->getManager();

                // Zapis.
                $em->persist($user);
                $em->flush();

                // Ustawienie komunikatu o zmianie hasła.
                $this->addFlash('user.reset.success', 'Hasło zostało zmienione.');

                // Zapisanie logu o zresetowaniu hasła.
                $this->get('wi.front.user.logger')->create(17, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'Password has been resetted.',
                ]);
            } else {
                // Zapisanie logu o błędzie podczas zresetowania hasła.
                $this->get('wi.front.user.logger')->create(18, [
                    'user_id' => $user->getId(),
                    'ip' => $request->server->get('REMOTE_ADDR'),
                    'host' => $request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')),
                    'descriptions' => 'An error occurred while password reset.',
                ]);
            }
        }

        return $this->render($this->getTemplate('Users/Security/reset_password'), [
            'form' => $form->createView(),
        ]);
    }
}
