<?php

namespace Wi\Front\CoreBundle\Twig\Extension;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\RouterInterface;
use Wi\Admin\CoreBundle\Service\Config;
use Wi\Admin\CoreBundle\Service\Seo;
use Wi\Admin\CoreBundle\Service\Cookie;
use Wi\Admin\CoreBundle\Service\MenuAdmin;
use Wi\Front\CoreBundle\Utils\Slugger;

/**
 * Front extension for twig.
 *
 * @author      Jakub Nowak <mr9nowak@gmail.com>
 * @copyright   2017 Jakub Nowak
 * @link        http://www.jakubnowak.com.pl/
 * @version     1.0.0
 */
class FrontExtension extends \Twig_Extension
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * @var RequestStack
     */
    private $requestStack;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var Seo
     */
    private $seo;

    /**
     * @var Cookie
     */
    private $cookie;

    /**
     * @var Slugger
     */
    private $slugger;

    /**
     * @var MenuAdmin
     */
    private $menuAdmin;


    // ------------------------------------------------------------
    // Methods.
    // ------------------------------------------------------------

    /**
     * Constructor.
     *
     * @param Config $config
     * @param EntityManagerInterface $em
     * @param RequestStack $requestStack
     * @param RouterInterface $router
     * @param Seo $seo
     * @param Slugger $slugger
     */
    public function __construct(Config $config, EntityManagerInterface $em, RequestStack $requestStack, RouterInterface $router, Seo $seo, Slugger $slugger, Cookie $cookie, MenuAdmin $menuAdmin)
    {
        $this->config = $config;
        $this->em = $em;
        $this->requestStack = $requestStack;
        $this->router = $router;
        $this->seo = $seo;
        $this->cookie = $cookie;
        $this->slugger = $slugger;
        $this->menuAdmin = $menuAdmin;
    }

    /**
     * Pobiera aukcje z możliwością zawężenia wyników do kategorii i edycji.
     * Aukcje są sortowane według czasu zakończenia od najwcześniej kończącej się
     * do najpóźniej kończącej.
     *
     * @param   int     $category_id  Kategoria, z której mają zostać pobrane aukcje.
     * @param   int     $edition_id   Edycja, z której mają zostać pobrane aukcje.
     * @param   int     $limit        Limit celów do pobrania. Domyślnie - 3.
     * @return  array
     */
    public function getAuctions($category_id = null, $edition_id = null, $limit = 3)
    {
        return $this->em
            ->getRepository('WiBidbeeMarketBundle:Auction')
            ->findByCategoryEditionWithLimit($category_id, $edition_id, $limit)
        ;
    }

    /**
     * Pobiera cele fundacji na podstawie edycji z zachowaniem limitu.
     * Pobierane cele są najpierw sortowane malejąco po dacie utworzenia
     * następnie wybierane są najnowsze.
     *
     * @param   int     $edition_id   Edycja, z której mają zostać pobrane cele.
     * @param   int     $limit        Limit celów do pobrania. Domyślnie - 3.
     * @return  array
     */
    public function getGoals($edition_id, $limit = 3)
    {
        return $this->em
            ->getRepository('WiBidbeeFoundationBundle:Goal')
            ->findByEditionWithLimit($edition_id, $limit)
        ;
    }

    /**
     * Pobiera aktualności na podstawie kategorii ograniczone do podanego limitu.
     *
     * @param   int     $category_id  Kategoria, z której mają zostać pobrane aktualności.
     *                                Jeśli 'null' to pobiera ze wszystkich kategorii.
     * @param   int     $limit        Limit aktualności do pobrania. Domyślnie - 3.
     * @return  array
     */
    public function getNews($category_id = null, $limit = 3)
    {
        return $this->em
            ->getRepository('WiAdminNewsBundle:News')
            ->findByCategoryWithLimit("N", $category_id, $limit)
        ;
    }

    /**
     * Pobiera aktualności na podstawie kategorii bez ograniczenia limitu, sortowane według daty publikacji.
     *
     * @param   int     $category_id  Kategoria, z której mają zostać pobrane aktualności.
     *                                Jeśli 'null' to pobiera ze wszystkich kategorii.
     * @return  array
     */
    public function getNewsNoLimit($category_id = null)
    {
        return $this->em
            ->getRepository('WiAdminNewsBundle:News')
            ->findByCategoryNotLimit($category_id)
            ;
    }

    /**
     * Pobiera statystyki newsów i stron.
     *
     * @return  array
     */
    public function getStats()
    {
        $static = [];
        // Utworzenie statystyk newsów.
        $newsCount = $this->em->getRepository('WiAdminNewsBundle:News')->findByStatus('A');
        $categoryCount = $this->em->getRepository('WiAdminNewsBundle:Category')->findByStatus('A');
        $static['categoryCount'] = count($categoryCount);
        $static['newsCount'] = count($newsCount);

        // Utworzenie statystyk stron.
        $pagesCount = $this->em->getRepository('WiAdminPageBundle:Page')->findByStatus('A');
        $groupCount = $this->em->getRepository('WiAdminPageBundle:PageGroup')->findByStatus('A');
        $static['groupCount'] = count($groupCount);
        $static['pagesCount'] = count($pagesCount);

        return $static;
    }

    /**
     * Slugify.
     *
     * @param string $string
     * @return string
     */
    public function slugify($string)
    {
        return $this->slugger->slugify($string);
    }

    // ------------------------------------------------------------
    // SEO functions.
    // ------------------------------------------------------------

    /**
     * Get meta canonical for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaCanonical($object)
    {
        switch (get_class($object)) {
            case 'Wi\Admin\PageBundle\Entity\Page':
                $canonical = $this->router->generate('wi_front_page_index', [
                    'id' => $object->getId(),
                    'slug' => $this->slugger->slugify($object->getTitle()),
                ], 0);

                if (! empty($object->getMetaCanonical())) {
                    $canonical = $object->getMetaCanonical();
                }
                break;

            case 'Wi\Admin\NewsBundle\Entity\News':
                $canonical = $this->router->generate('wi_front_news_show', [
                    'id' => $object->getId(),
                    'slug' => $this->slugger->slugify($object->getTitle()),
                ], 0);

                if (! empty($object->getMetaCanonical())) {
                    $canonical = $object->getMetaCanonical();
                }
                break;

            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
                $canonical = $this->router->generate('wi_front_foundation_show', [
                    'id' => $object->getId(),
                    'slug' => $this->slugger->slugify($object->getName()),
                ], 0);

                if (! empty($object->getMetaCanonical())) {
                    $canonical = $object->getMetaCanonical();
                }
                break;

            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                $canonical = $this->router->generate('wi_front_market_show', [
                    'id' => $object->getId(),
                    'slug' => $this->slugger->slugify($object->getTitle()),
                ], 0);

                if (! empty($object->getMetaCanonical())) {
                    $canonical = $object->getMetaCanonical();
                }
                break;

            default:
                $canonical = false;
                break;
        }

        return $canonical;
    }

    /**
     * Get meta description for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaDescription($object)
    {
        switch (get_class($object)) {
            case 'Wi\Admin\PageBundle\Entity\Page':
            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                $description = preg_replace(
                    '/(.*)\s/',
                    '$1',
                    mb_substr(strip_tags($object->getContent()), 0, 500)
                );

                if (! empty($object->getMetaDescription())) {
                    $description = $object->getMetaDescription();
                }
                break;

            case 'Wi\Admin\NewsBundle\Entity\News':
                $description = strip_tags($object->getShortcut());

                if (empty($description)) {
                    $description = preg_replace(
                        '/(.*)\s/',
                        '$1',
                        mb_substr(strip_tags($object->getContent()), 0, 500)
                    );
                }

                if (! empty($object->getMetaDescription())) {
                    $description = $object->getMetaDescription();
                }
                break;

            default:
                $description = false;
                break;
        }

        return $description;
    }

    /**
     * Get meta image for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaImage($object)
    {
        $image = false;

        // Get request.
        $request = $this->requestStack->getCurrentRequest();

        // Set media url.
        $mediaUrl = $request->getScheme() . '://' . $request->getHttpHost() .
            $request->getBasePath() . '/media/assets/'
        ;

        switch (get_class($object)) {
            case 'Wi\Admin\PageBundle\Entity\Page':
                $image = $object->getMetaImage();
                break;

            case 'Wi\Admin\NewsBundle\Entity\News':
                if (! is_null($object->getImageUrl())) {
                    $image = $mediaUrl . 'News/' . $object->getImageUrl();
                }

                if (! empty($object->getMetaImage())) {
                    $image = $object->getMetaImage();
                }
                break;

            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
                if (! is_null($object->getImage())) {
                    $image = $mediaUrl . 'Goal/' . $object->getImage();
                }

                if (! empty($object->getMetaImage())) {
                    $image = $object->getMetaImage();
                }
                break;

            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                if (! is_null($object->getImage())) {
                    $image = $mediaUrl . 'Auction/' . $object->getImage();
                }

                if (! empty($object->getMetaImage())) {
                    $image = $object->getMetaImage();
                }
                break;

            default:
                $image = false;
                break;
        }

        return $image;
    }

    /**
     * Get meta keywords for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaKeywords($object)
    {
        switch (get_class($object)) {
            case 'Wi\Admin\NewsBundle\Entity\News':
            case 'Wi\Admin\PageBundle\Entity\Page':
            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                $keywords = $object->getMetaKeywords();
                break;

            default:
                $keywords = false;
                break;
        }

        return $keywords;
    }

    /**
     * Get meta title for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaTitle($object)
    {
        switch (get_class($object)) {
            case 'Wi\Admin\NewsBundle\Entity\News':
            case 'Wi\Admin\PageBundle\Entity\Page':
            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                $title = $object->getTitle();

                if (! empty($object->getMetaTitle())) {
                    $title = $object->getMetaTitle();
                }
                break;

            default:
                $title = false;
                break;
        }

        return $title;
    }

    /**
     * Get meta type for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaType($object)
    {
        switch (get_class($object)) {
            case 'Wi\Admin\PageBundle\Entity\Page':
                $type = $this->config->get('og.pageType');

                if (! empty($object->getMetaType())) {
                    $type = $object->getMetaType();
                }
                break;

            case 'Wi\Admin\NewsBundle\Entity\News':
                $type = $this->config->get('og.newsType');

                if (! empty($object->getMetaType())) {
                    $type = $object->getMetaType();
                }
                break;

            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
                $type = $this->config->get('og.goalType');

                if (! empty($object->getMetaType())) {
                    $type = $object->getMetaType();
                }
                break;

            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                $type = $this->config->get('og.auctionType');

                if (! empty($object->getMetaType())) {
                    $type = $object->getMetaType();
                }
                break;

            default:
                $type = false;
                break;
        }

        return $type;
    }

    // -------------------------------------------------------------------------

    /**
     * Pobiera wyrenderowane metadane.
     *
     * @param object $obj
     * @return string
     */
    public function getMeta($obj = null)
    {
        return $this->seo->getMeta($obj);
    }

    /**
     * Pobiera adres kanoniczny.
     *
     * @param object $obj
     * @return string
     */
    public function getCanonical($obj = null) : string
    {
        return $this->seo->getCanonical($obj);
    }

    /**
     * Pobiera wyrenderowane metadane.
     *
     * @param object $obj
     * @return string
     */
    public function getMetaArrayValues($obj = null)
    {
        return $this->seo->getMetaArrayValues($obj);
    }

    /**
     * Pobiera wyrenderowane komunikaty o cookies.
     *
     * @return string
     */
    public function getCookie()
    {
        return $this->cookie->getCookie();
    }

    /**
     * Pobiera elementy menu administracyjnego.
     *
     * @param int $group_id
     * @return array
     */
    public function getAdminMenu($group_id, $attr, $menuIsActive)
    {
        return $this->menuAdmin->getMenu($group_id, $attr, $menuIsActive);
    }

    /**
     * Pobiera bloki HTML.
     *
     * @param int $block_id
     * @param int $limit
     * @return array
     */
    public function getBlock($block_id, $limit = null)
    {
        return $this->em->getRepository('WiAdminBlockBundle:ItemHTML')->findActivByBlockId($block_id, $limit);
    }

    /**
     * Pobiera konfigurację
     *
     * @param string $key
     * return string
     */
    public function getConfig($key)
    {
        return $this->config->get($key);
    }

    /**
     * Pobiera bannery.
     *
     * @param int $bannerSet_id
     * @return array
     */
    public function getBanners($bannerSet_id)
    {
        return $this->em->getRepository('WiAdminBannerBundle:Banner')->findActivByBannerSetId($bannerSet_id);
    }

    // ------------------------------------------------------------
    // Inherit functions.
    // ------------------------------------------------------------

    /**
     * {@inheritdoc}
     */
    public function getFunctions()
    {
        return [
            new \Twig_SimpleFunction('slugify', [$this, 'slugify']),
            new \Twig_SimpleFunction('getAuctions', [$this, 'getAuctions']),
            new \Twig_SimpleFunction('getGoals', [$this, 'getGoals']),
            new \Twig_SimpleFunction('getNews', [$this, 'getNews']),
            new \Twig_SimpleFunction('getNewsNoLimit', [$this, 'getNewsNoLimit']),
            new \Twig_SimpleFunction('getStats', [$this, 'getStats']),
            new \Twig_SimpleFunction('getMetaCanonical', [$this, 'getMetaCanonical']),
            new \Twig_SimpleFunction('getMetaDescription', [$this, 'getMetaDescription']),
            new \Twig_SimpleFunction('getMetaImage', [$this, 'getMetaImage']),
            new \Twig_SimpleFunction('getMetaKeywords', [$this, 'getMetaKeywords']),
            new \Twig_SimpleFunction('getMetaTitle', [$this, 'getMetaTitle']),
            new \Twig_SimpleFunction('getMetaType', [$this, 'getMetaType']),
            new \Twig_SimpleFunction('getMeta', [$this, 'getMeta'], ['is_safe' => ['html']]),
            new \Twig_SimpleFunction('getCanonical', [$this, 'getCanonical']),
            new \Twig_SimpleFunction('getMetaArrayValues', [$this, 'getMetaArrayValues']),
            new \Twig_SimpleFunction('getCookie', [$this, 'getCookie']),
            new \Twig_SimpleFunction('getAdminMenu', [$this, 'getAdminMenu']),
            new \Twig_SimpleFunction('getBlock', [$this, 'getBlock']),
            new \Twig_SimpleFunction('getConfig', [$this, 'getConfig']),
            new \Twig_SimpleFunction('getBanners', [$this, 'getBanners']),
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return 'app_front';
    }
}
