<?php

namespace Wi\Front\CoreBundle\Templating\Helper;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Templating\Helper\Helper;
use Wi\Admin\ContactBundle\Entity\Form;
use Wi\Admin\CoreBundle\Service\Config;
use Wi\Admin\CoreBundle\Utils\Upload;
use Wi\Admin\NewsBundle\Entity\News;
use Wi\Admin\NewsBundle\Entity\Category;
use Wi\Front\CoreBundle\Utils\Slugger;
use Wi\Admin\CoreBundle\Service\MenuAdmin;
use Wi\Admin\CoreBundle\Service\Seo;

/**
 * Front Helper.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class FrontHelper extends Helper
{
    /**
     * @var Config
     */
    private $config;

    /**
     * Container Interface object.
     *
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * @var RequestStack
     */
    private $requestStack;

    /**
     * @var RequestContext
     */
    private $requestContext;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var Seo
     */
    private $seo;

    /**
     * @var Slugger
     */
    private $slugger;

    /**
     * @var Upload
     */
    private $upload;

    /**
     * @var MenuAdmin
     */
    private $menuAdmin;

    /**
     * Construct.
     *
     * @param Config $config
     * @param ContainerInterface $container
     * @param EntityManagerInterface $em
     * @param Slugger $slugger
     * @param RequestStack $requestStack
     * @param RequestContext $requestContext
     * @param Seo $seo
     */
    public function __construct(Config $config, ContainerInterface $container, EntityManagerInterface $em, Slugger $slugger, RequestStack $requestStack, Upload $upload, RequestContext $requestContext = null, MenuAdmin $menuAdmin, Seo $seo)
    {
        $this->config = $config;
        $this->container = $container;
        $this->em = $em;
        $this->router = $container->get('router');
        $this->slugger = $slugger;
        $this->upload = $upload;
        $this->requestStack = $requestStack;
        $this->menuAdmin = $menuAdmin;
        $this->seo = $seo;
    }

    /**
     * Funkcja zaczerpnięta z Twig-a.
     *
     * Returns the absolute URL for the given absolute or relative path.
     * This method returns the path unchanged if no request is available.
     *
     * @param string $path The path
     * @return string The absolute URL
     * @see \Symfony\Bridge\Twig\Extension\HttpFoundationExtension::generateAbsoluteUrl()
     */
    public function absoluteUrl($path)
    {
        if (false !== strpos($path, '://') || '//' === substr($path, 0, 2)) {
            return $path;
        }

        if (!$request = $this->requestStack->getMasterRequest()) {
            if (null !== $this->requestContext && '' !== $host = $this->requestContext->getHost()) {
                $scheme = $this->requestContext->getScheme();
                $port = '';

                if ('http' === $scheme && 80 != $this->requestContext->getHttpPort()) {
                    $port = ':'.$this->requestContext->getHttpPort();
                } elseif ('https' === $scheme && 443 != $this->requestContext->getHttpsPort()) {
                    $port = ':'.$this->requestContext->getHttpsPort();
                }

                if ('#' === $path[0]) {
                    $queryString = $this->requestContext->getQueryString();
                    $path = $this->requestContext->getPathInfo().($queryString ? '?'.$queryString : '').$path;
                } elseif ('?' === $path[0]) {
                    $path = $this->requestContext->getPathInfo().$path;
                }

                if ('/' !== $path[0]) {
                    $path = rtrim($this->requestContext->getBaseUrl(), '/').'/'.$path;
                }

                return $scheme.'://'.$host.$port.$path;
            }

            return $path;
        }

        if ('#' === $path[0]) {
            $path = $request->getRequestUri().$path;
        } elseif ('?' === $path[0]) {
            $path = $request->getPathInfo().$path;
        }

        if (!$path || '/' !== $path[0]) {
            $prefix = $request->getPathInfo();
            $last = strlen($prefix) - 1;
            if ($last !== $pos = strrpos($prefix, '/')) {
                $prefix = substr($prefix, 0, $pos).'/';
            }

            return $request->getUriForPath($prefix.$path);
        }

        return $request->getSchemeAndHttpHost().$path;
    }

    /**
     * Osadzanie zdjęć w szablonach email.
     *
     * @param string $path
     * @param \Swift_Message $message
     * @return string
     */
    public function embedImage($path, $message)
    {
        return $message->embed(\Swift_Image::fromPath($this->absoluteUrl($path)));
    }

    /**
     * Pobiera aukcje z możliwością zawężenia wyników do kategorii i edycji.
     * Aukcje są sortowane według czasu zakończenia od najwcześniej kończącej się
     * do najpóźniej kończącej.
     *
     * @param   int     $category_id  Kategoria, z której mają zostać pobrane aukcje.
     * @param   int     $edition_id   Edycja, z której mają zostać pobrane aukcje.
     * @param   int     $limit        Limit celów do pobrania. Domyślnie - 3.
     * @return  array
     */
    public function getAuctions($category_id = null, $edition_id = null, $limit = 3)
    {
        return $this->em
            ->getRepository('WiBidbeeMarketBundle:Auction')
            ->findByCategoryEditionWithLimit($category_id, $edition_id, $limit)
        ;
    }

    /**
     * Pobiera cele fundacji na podstawie edycji z zachowaniem limitu.
     * Pobierane cele są najpierw sortowane malejąco po dacie utworzenia
     * następnie wybierane są najnowsze.
     *
     * @param   int     $edition_id   Edycja, z której mają zostać pobrane cele.
     * @param   int     $limit        Limit celów do pobrania. Domyślnie - 3.
     * @return  array
     */
    public function getGoals($edition_id, $limit = 3)
    {
        return $this->em
            ->getRepository('WiBidbeeFoundationBundle:Goal')
            ->findByEditionWithLimit($edition_id, $limit)
        ;
    }

    /**
     * Pobiera aktualności na podstawie kategorii ograniczone do podanego limitu.
     *
     * @param   int     $category_id  Kategoria, z której mają zostać pobrane aktualności.
     *                                Jeśli 'null' to pobiera ze wszystkich kategorii.
     * @param   int     $limit        Limit aktualności do pobrania. Domyślnie - 3.
     * @return  array
     */
    public function getNews($category_id = null, $limit = 3)
    {
        return $this->em
            ->getRepository('WiAdminNewsBundle:News')
            ->findByCategoryWithLimit($category_id, $limit)
        ;
    }

    /**
     * Statystyki. Liczba newsów, kategorii, grup stron, stron, formularzy,
     * wiadomości wszystkich i nieprzeczytanych.
     *
     * @return array
     */
    public function getStats()
    {
        return [
            'categoryCount' => $this->em->getRepository('WiAdminNewsBundle:Category')->countActive(Category::TYPE_NEWS),
            'categoryBlogCount' => $this->em->getRepository('WiAdminNewsBundle:Category')->countActive(Category::TYPE_BLOG),
            'groupCount' => $this->em->getRepository('WiAdminPageBundle:PageGroup')->countActive(),
            'newsCount' => $this->em->getRepository('WiAdminNewsBundle:News')->countActive(News::TYPE_NEWS),
            'blogCount' => $this->em->getRepository('WiAdminNewsBundle:News')->countActive(News::TYPE_BLOG),
            'pagesCount' => $this->em->getRepository('WiAdminPageBundle:Page')->countActive(),
            'formCount' => $this->em->getRepository('WiAdminContactBundle:Form')->countActive(),
            'formCountContactAnswer' => $this->getAnswerRepo()->countByFormType(Form::TYPE_CONTACT),
            'formCountContactNewAnswer' => $this->getAnswerRepo()->countUnreadedByFormType(Form::TYPE_CONTACT),
            'formCountLeadAnswer' => $this->getAnswerRepo()->countByFormType(Form::TYPE_LEAD),
            'formCountLeadNewAnswer' => $this->getAnswerRepo()->countUnreadedByFormType(Form::TYPE_LEAD),
            'formCountGroup' => $this->getAnswerRepo()->countGroupByForm(),
            'formCountUnreadedGroup' => $this->getAnswerRepo()->countUnreadedGroupByForm(),
            'commentCount' => $this->em->getRepository('WiAdminNewsBundle:Comment')->countActive(),
            'commentNewsCount' => $this->em->getRepository('WiAdminNewsBundle:Comment')->countPostActive(),
        ];
    }

    /**
     * Slugify.
     *
     * @param   string $string
     * @return  string
     */
    public function slugify($string)
    {
        return $this->slugger->slugify($string);
    }

    // ------------------------------------------------------------
    // SEO functions.
    // ------------------------------------------------------------

    /**
     * Pobiera wyrenderowane metadane.
     *
     * @param object $obj
     * @return string
     */
    public function getMetaArrayValues($obj = null)
    {
        return $this->seo->getMetaArrayValues($obj);
    }

    /**
     * Get meta canonical for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaCanonical($object)
    {
        switch (get_class($object)) {
            case 'Wi\Admin\PageBundle\Entity\Page':
                $canonical = $this->router->generate('wi_front_page_index', [
                    'id' => $object->getId(),
                    'slug' => $this->slugger->slugify($object->getTitle()),
                ], 0);

                if (! empty($object->getMetaCanonical())) {
                    $canonical = $object->getMetaCanonical();
                }
                break;

            case 'Wi\Admin\NewsBundle\Entity\News':
                $canonical = $this->router->generate('wi_front_news_show', [
                    'id' => $object->getId(),
                    'slug' => $this->slugger->slugify($object->getTitle()),
                ], 0);

                if (! empty($object->getMetaCanonical())) {
                    $canonical = $object->getMetaCanonical();
                }
                break;

            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
                $canonical = $this->router->generate('wi_front_foundation_show', [
                    'id' => $object->getId(),
                    'slug' => $this->slugger->slugify($object->getName()),
                ], 0);

                if (! empty($object->getMetaCanonical())) {
                    $canonical = $object->getMetaCanonical();
                }
                break;

            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                $canonical = $this->router->generate('wi_front_market_show', [
                    'id' => $object->getId(),
                    'slug' => $this->slugger->slugify($object->getTitle()),
                ], 0);

                if (! empty($object->getMetaCanonical())) {
                    $canonical = $object->getMetaCanonical();
                }
                break;

            default:
                $canonical = false;
                break;
        }

        return $canonical;
    }

    /**
     * Get meta description for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaDescription($object)
    {
        switch (get_class($object)) {
            case 'Wi\Admin\PageBundle\Entity\Page':
            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                $description = preg_replace(
                    '/(.*)\s/',
                    '$1',
                    mb_substr(strip_tags($object->getContent()), 0, 500)
                );

                if (! empty($object->getMetaDescription())) {
                    $description = $object->getMetaDescription();
                }
                break;

            case 'Wi\Admin\NewsBundle\Entity\News':
                $description = strip_tags($object->getShortcut());

                if (empty($description)) {
                    $description = preg_replace(
                        '/(.*)\s/',
                        '$1',
                        mb_substr(strip_tags($object->getContent()), 0, 500)
                    );
                }

                if (! empty($object->getMetaDescription())) {
                    $description = $object->getMetaDescription();
                }
                break;

            default:
                $description = false;
                break;
        }

        return $description;
    }

    /**
     * Get meta image for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaImage($object)
    {
        $image = false;

        // Get request.
        $request = $this->requestStack->getCurrentRequest();

        // Set media url.
        $mediaUrl = $request->getScheme() . '://' . $request->getHttpHost() .
            $request->getBasePath() . '/media/assets/'
        ;

        switch (get_class($object)) {
            case 'Wi\Admin\PageBundle\Entity\Page':
                $image = $object->getMetaImage();
                break;

            case 'Wi\Admin\NewsBundle\Entity\News':
                if (! is_null($object->getImageUrl())) {
                    $image = $mediaUrl . 'News/' . $object->getImageUrl();
                }

                if (! empty($object->getMetaImage())) {
                    $image = $object->getMetaImage();
                }
                break;

            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
                if (! is_null($object->getImage())) {
                    $image = $mediaUrl . 'Goal/' . $object->getImage();
                }

                if (! empty($object->getMetaImage())) {
                    $image = $object->getMetaImage();
                }
                break;

            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                if (! is_null($object->getImage())) {
                    $image = $mediaUrl . 'Auction/' . $object->getImage();
                }

                if (! empty($object->getMetaImage())) {
                    $image = $object->getMetaImage();
                }
                break;

            default:
                $image = false;
                break;
        }

        return $image;
    }

    /**
     * Get meta keywords for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaKeywords($object)
    {
        switch (get_class($object)) {
            case 'Wi\Admin\NewsBundle\Entity\News':
            case 'Wi\Admin\PageBundle\Entity\Page':
            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                $keywords = $object->getMetaKeywords();
                break;

            default:
                $keywords = false;
                break;
        }

        return $keywords;
    }

    /**
     * Get meta title for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaTitle($object)
    {
        switch (get_class($object)) {
            case 'Wi\Admin\NewsBundle\Entity\News':
            case 'Wi\Admin\PageBundle\Entity\Page':
            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                $title = $object->getTitle();

                if (! empty($object->getMetaTitle())) {
                    $title = $object->getMetaTitle();
                }
                break;

            default:
                $title = false;
                break;
        }

        return $title;
    }

    /**
     * Get meta type for object.
     *
     * @param   object $object
     * @return  string
     */
    public function getMetaType($object)
    {
        switch (get_class($object)) {
            case 'Wi\Admin\PageBundle\Entity\Page':
                $type = $this->config->get('og.pageType');

                if (! empty($object->getMetaType())) {
                    $type = $object->getMetaType();
                }
                break;

            case 'Wi\Admin\NewsBundle\Entity\News':
                $type = $this->config->get('og.newsType');

                if (! empty($object->getMetaType())) {
                    $type = $object->getMetaType();
                }
                break;

            case 'Wi\Bidbee\FoundationBundle\Entity\Goal':
                $type = $this->config->get('og.goalType');

                if (! empty($object->getMetaType())) {
                    $type = $object->getMetaType();
                }
                break;

            case 'Wi\Bidbee\MarketBundle\Entity\Auction':
                $type = $this->config->get('og.auctionType');

                if (! empty($object->getMetaType())) {
                    $type = $object->getMetaType();
                }
                break;

            default:
                $type = false;
                break;
        }

        return $type;
    }

    /**
     * Pobiera elementy menu administracyjnego.
     *
     * @param int $group_id
     * @return array
     */
    public function getAdminMenu($group_id, $attr, $menuIsActive)
    {
        return $this->menuAdmin->getMenu($group_id, $attr, $menuIsActive);
    }

    /**
     * Format bytes to kilobytes, megabytes, gigabytes, terabytes.
     *
     * @param   int     $bytes
     * @return  string
     */
    public function formatBytes($bytes, $dec = 2)
    {
        return $this->upload->formatBytes($bytes, $dec);
    }

    /**
     * Pobiera adres kanoniczny.
     *
     * @param object $obj
     * @return string
     */
    public function getCanonical($obj = null) : string
    {
        return $this->seo->getCanonical($obj);
    }

    // ------------------------------------------------------------
    // Inherit functions.
    // ------------------------------------------------------------

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return 'front';
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Zwraca obiekt repozytorium odpowiedzi.
     *
     * @return AnswerRepository
     */
    private function getAnswerRepo()
    {
        return $this->em->getRepository('WiAdminContactBundle:Answer');
    }
}
