<?php

namespace Wi\Front\ContactBundle\Controller;

use DeviceDetector\DeviceDetector;
use DeviceDetector\Parser\Device\DeviceParserAbstract;
use Wi\Front\CoreBundle\Core\Controller\FrontController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Wi\Admin\ContactBundle\Entity\Answer;
use Wi\Admin\ContactBundle\Entity\AnswerField;
use Wi\Admin\ContactBundle\Entity\Form;
use Wi\Admin\ContactBundle\Entity\Form\FormFieldTypeInterface;
use Wi\Admin\ContactBundle\Service\FormGenerator;
use Wi\Admin\ContactBundle\Service\Mailer;
use Wi\Admin\CoreBundle\Service\Seo;
use Wi\Admin\NewsLetterBundle\Service\RecipientManager;
use Wi\Admin\NewsLetterBundle\Entity\Recipients;
use Wi\Admin\NewsLetterBundle\Service\HashRecipientGroup;

/**
 * Contact controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class ContactController extends FrontController
{
    private $autoreply = false;

    /**
     * Kontakt.
     *
     * @param FormGenerator $formGenerator
     * @param Form|null $contactForm
     * @return Response
     */
    public function indexAction(FormGenerator $formGenerator, Form $contactForm = null)
    {
        if (is_null($contactForm)) {
            $contactForm = $this->getFormRepository()->findFirst();
        }

        $page = null;

        $em = $this->getDoctrine()->getManager();
        $em->getFilters()->disable('softdeleteable');

        if (! is_null($contactForm)) {
            if (! is_null($contactForm->getPage())) {
                if (is_null($contactForm->getPage()->getDateDeleted())) {
                    $page = $contactForm->getPage();
                    $breadcrumbsName = $page->getTitle();
                }
            } else {
                $breadcrumbsName = $contactForm->getName();
            }
            // Breadcrumbs.
            $this->breadcrumbs->addItem($breadcrumbsName, $this->generateUrl("wi_front_contact_index_2", ['id'=>$contactForm->getId(), 'slug'=>$this->slugify($contactForm->getName())]));
        } else {
            // Breadcrumbs.
            $this->breadcrumbs->addItem("Kontakt", $this->generateUrl("wi_front_contact_index"));
        }
        $em->getFilters()->enable('softdeleteable');

        return $this->render($this->getTemplate('Contact/index'), [
            'contactForm' => $contactForm,
            'page' => $page,
            'form' => $formGenerator->generate($contactForm),
        ]);
    }

    /**
     * Odbiór wiadomości.
     *
     * @param FormGenerator $formGenerator
     * @param Request $request
     * @param Mailer $mailer
     * @return RedirectResponse|JsonResponse
     */
    public function receiverAction(FormGenerator $formGenerator, Request $request, Mailer $mailer, Seo $seo)
    {
        $post = $request->request;
        $json = boolval($request->query->getInt('json'));
        $form = $this->getFormRepository()->find($post->getInt('form_id'));

        // Nie znaleziono formularza.
        if (is_null($form)) {
            if ($json) {
                return new JsonResponse([
                    'status' => 'ERROR',
                    'message' => 'Nie udało się wysłać wiadomości, formularz nie istnieje.',
                ]);
            }

            $this->addFlash('contact.error', 'Nie udało się wysłać wiadomości.');

            return $this->redirectToRoute('wi_front_contact_index');
        }

        // Zabezpieczenie antyspamowe nie jest puste (to jest SPAM).
        if (! empty($post->get('email_repeat_123_'))) {
            if ($json) {
                return new JsonResponse([
                    'status' => 'OK',
                    'message' => 'Wiadomość została wysłana.',
                ]);
            }

            $this->addFlash('contact.error', 'Wiadomość została wysłana.');

            return $this->redirect($seo->getMetaArrayValues($form)['canonical']);
        }

        if ($form->getRecaptcha()) {
            if (! $this->verifyRecaptcha($formGenerator, $post->get('g-recaptcha-response'))) {
                if ($json) {
                    return new JsonResponse([
                        'status' => 'ERROR',
                        'message' => 'Potwierdź, że nie jesteś robotem.',
                    ]);
                }

                $this->addFlash('contact.error', 'Potwierdź, że nie jesteś robotem.');

                return $this->redirectToRoute('wi_front_contact_index');
            }
        }

        // Rozpoznawanie urządzenia.
        $dd = new DeviceDetector($request->server->get('HTTP_USER_AGENT'));
        $dd->parse();

        $answer = new Answer();
        $answer
            ->setForm($form)
            ->setMobile($dd->isMobile())
            ->setIp($request->server->get('REMOTE_ADDR'))
            ->setHost($request->server->get('REMOTE_HOST') ?: gethostbyaddr($request->server->get('REMOTE_ADDR')))
            ->setAgent($request->server->get('HTTP_USER_AGENT'))
            ->setStatus($this->getNewStatus())
        ;

        foreach ($form->getFields() as $field) {
            if (! is_null($value = $post->get($field->getName()))) {
                $answer->addField($this->prepareAnswerField($field, $value));
            }
        }

        $em = $this->getDoctrine()->getManager();
        $em->persist($answer);
        $em->flush($answer);

        $res = $mailer->sendContactFormMessage($answer);

        if ($this->autoreply && $form->getAutoreply()) {
            $mailer->sendAutoreply($form, $this->autoreply);
        }

        if (! $res) {
            if ($json) {
                return new JsonResponse([
                    'status' => 'ERROR',
                    'message' => 'Nie udało się wysłać wiadomości.',
                ]);
            }

            $this->addFlash('contact.error', 'Nie udało się wysłać wiadomości.');

            return $this->redirectToRoute('wi_front_contact_index');
        }


        if ($json) {
            return new JsonResponse([
                'status' => 'OK',
                'message' => 'Wiadomość została wysłana.',
            ]);
        }

        $this->addFlash('contact.success', 'Wiadomość została wysłana.');

        return $this->redirect($seo->getMetaArrayValues($form)['canonical']);
    }

    /**
     *
     * @param RecipientManager $recipientManager
     * @param HashRecipientGroup $hasher
     * @param Request $request
     * @return JsonResponse
     */
    public function newsletterReceiverAction(RecipientManager $recipientManager, HashRecipientGroup $hasher, Request $request)
    {
        if (! empty($request->request->get('email_repeat'))) {
            return new JsonResponse([
                'status' => true,
            ]);
        }

        $result = $recipientManager->signUpRecipient([
            'email' => trim(strip_tags($request->request->get('email'))),
            'name' => trim(strip_tags($request->request->get('name'))),
            'group' => $hasher->decode($request->request->get('group')),
        ]);

        if ($result) {
            $recipientManager->generateActiveEmail();
        }

        return new JsonResponse([
            'status' => $result,
        ]);
    }

    public function newsletterConfirmAction(Request $request, Recipients $recipient)
    {
        $recipient
            ->setConsent(true)
            ->setHash(null)
        ;

        $em = $this->getDoctrine()->getManager();
        $em->persist($recipient);
        $em->flush($recipient);

        return $this->render('WiFrontContactBundle:Contact:newsletter_confirm.html.twig', [
        ]);
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Pobiera repozytorium formularzy kontaktowych.
     *
     * @return FormRepository
     */
    private function getFormRepository()
    {
        return $this
            ->getDoctrine()
            ->getManager()
            ->getRepository('WiAdminContactBundle:Form')
        ;
    }

    private function prepareAnswerField($field, $value)
    {
        $answerField = new AnswerField();
        $answerField->setField($field);

        switch ($field->getFormFieldType()->getType()) {
            case FormFieldTypeInterface::TEXTAREA:
                $answerField->setFieldTextValue($value);
                break;
            case FormFieldTypeInterface::INPUT_EMAIL:
                $this->autoreply = $value;
                // no break
            default:
                $answerField->setFieldValue($value);
                break;
        }

        return $answerField;
    }

    private function getNewStatus()
    {
        return $this->getDoctrine()->getManager()
            ->getRepository('WiAdminContactBundle:AnswerStatus')
            ->find(1)
        ;
    }

    /**
     * Weryfikacja recaptchy.
     *
     * @param FormGenerator $formGenerator
     * @param string response
     * @return
     */
    private function verifyRecaptcha(FormGenerator $formGenerator, $response)
    {
        return $formGenerator->verifyRecaptcha($response);
    }
}
