<?php

namespace Wi\Admin\UserBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;
use Symfony\Component\Security\Core\User\AdvancedUserInterface;
use Symfony\Component\Security\Core\User\EquatableInterface;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Validator\Constraints as Assert;
use Wi\Admin\NewsBundle\Entity\News;
use Wi\Admin\UserBundle\Entity\UserAvatar;
use Wi\Admin\UserBundle\Entity\BillingData;
use Wi\Admin\UserBundle\Entity\Log;
use Wi\Admin\UserBundle\Entity\Profile;
use Wi\Admin\UserBundle\Entity\UserInterest;

/**
 * User
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2018 WEBimpuls Sp. z o.o.
 *
 * @ORM\Table(
 *      name="users",
 *      uniqueConstraints={
 *          @ORM\UniqueConstraint(name="UQ_Users_Email",columns={"email"})
 *      }
 * )
 * @ORM\Entity(repositoryClass="Wi\Admin\UserBundle\Repository\UserRepository")
 * @ORM\EntityListeners({"Wi\Admin\UserBundle\EventListener\Entity\UserListener"})
 * @ORM\HasLifecycleCallbacks()
 * @UniqueEntity(fields="email", message="Konto o podanym adresie już istnieje.")
 */
class User implements AdvancedUserInterface, EquatableInterface, \Serializable
{
    /**
     * Status uzytkownika - aktywny.
     *
     * @const STATUS_ACTIVE
     */
    const STATUS_ACTIVE = 'A';

    /**
     * Status uzytkownika - zablokowany.
     *
     * @const STATUS_BLOCKED
     */
    const STATUS_BLOCKED = 'B';

    /**
     * Status uzytkownika - nieaktywny.
     *
     * @const STATUS_INACTIVE
     */
    const STATUS_INACTIVE = 'N';

    // -------------------------------------------------------------------------

    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="email", type="string", length=128, unique=true)
     * @Assert\NotBlank()
     * @Assert\Email()
     */
    private $email;

    /**
     * @var string
     *
     * @ORM\Column(name="password", type="string", length=256)
     * @Assert\NotBlank()
     */
    private $password;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateCreated", type="datetime")
     */
    private $dateCreated;

    /**
     * @var string
     *
     * @ORM\Column(name="hashCode", type="string", length=128, nullable=true)
     */
    private $hashCode;

    /**
     * @var string
     *
     * @ORM\Column(name="status", type="string", length=1, options={"default": "A", "fixed" = true})
     */
    private $status;

    /**
     * @var string
     *
     * @ORM\Column(name="allowNewsletter", type="boolean")
     */
    private $allowNewsletter;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\UserBundle\Entity\Profile", mappedBy="user")
     */
    private $profile;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\UserBundle\Entity\BillingData", mappedBy="user")
     */
    private $billingData;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\UserBundle\Entity\UserInterest", mappedBy="user")
     */
    private $userInterests;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\UserBundle\Entity\UserAvatar", mappedBy="user")
     */
    private $userAvatars;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\UserBundle\Entity\Log", mappedBy="user", orphanRemoval=true)
     */
    private $logs;

    /**
     * @var bool
     *
     * @Assert\IsTrue(
     *      message = "Akceptacja regulaminu jest wymagana.",
     *      groups={"register"}
     * )
     */
    private $confirm;

    /**
     * @var ArrayCollection
     *
     * @ORM\ManyToMany(targetEntity="\Wi\Admin\NewsBundle\Entity\News", inversedBy="users")
     * @ORM\JoinTable(name="user_favoriteNews",
     *      joinColumns={@ORM\JoinColumn(name="user_id", referencedColumnName="id")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="news_id", referencedColumnName="id")}
     * )
     */
    private $favoriteNews;

    /**
     * @var string
     *
     * @ORM\Column(name="image", type="string", length=256, nullable=true)
     */
    private $image;

    // ------------------------------------------------------------
    // Has Lifecycle Callbacks methods.
    // ------------------------------------------------------------

    /**
     * Funkcja wykonuje się podczas wstawiania nowego rekordu do bazy.
     *
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        // Ustawienie daty utworzenia.
        $this->dateCreated = new \DateTime();

        // Ustawienie statusu.
        if (is_null($this->status)) {
            $this->status = self::STATUS_ACTIVE;
        }

        // Ustawienie zezwolenia na wysyłanie maili.
        if (is_null($this->allowNewsletter)) {
            $this->allowNewsletter = false;
        }
    }

    // ------------------------------------------------------------
    // Advanced User Interface methods.
    // ------------------------------------------------------------

    public function getUsername()
    {
        return $this->email;
    }

    public function getRoles()
    {
        switch ($this->status) {
            case self::STATUS_ACTIVE: $roles = ['ROLE_USER', 'ROLE_USER_ACTIVE']; break;
            case self::STATUS_BLOCKED: $roles = ['ROLE_USER_BLOCKED']; break;
            case self::STATUS_INACTIVE: $roles = ['ROLE_USER_INACTIVE']; break;
            default : $roles = []; break;
        }

        return $roles;
    }

    public function getSalt()
    {
        return null;
    }

    public function eraseCredentials()
    {
    }

    public function isAccountNonExpired()
    {
        return true;
    }

    public function isAccountNonLocked()
    {
        return true;
    }

    public function isCredentialsNonExpired()
    {
        return true;
    }

    public function isEnabled()
    {
        return (in_array($this->status, [self::STATUS_ACTIVE, self::STATUS_INACTIVE]));
    }

    public function serialize()
    {
        return serialize([
            $this->id,
            $this->email,
            $this->password,
        ]);
    }

    public function unserialize($serialized)
    {
        list(
            $this->id,
            $this->email,
            $this->password
        ) = unserialize($serialized);
    }

    // ------------------------------------------------------------
    // Equatable Interface methods.
    // ------------------------------------------------------------

    public function isEqualTo(UserInterface $user)
    {
        if ($this->getId() == $user->getId()) {
            return true;
        }

        return false;
    }

    // ------------------------------------------------------------
    // Getters and setters methods.
    // ------------------------------------------------------------

    /**
     * Set id
     *
     * @param int $id
     * @return User
     */
    public function setId($id)
    {
        $this->id = $id;

        return $this;
    }

    /**
     * Get id
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set email
     *
     * @param string $email
     * @return User
     */
    public function setEmail($email)
    {
        $this->email = $email;

        return $this;
    }

    /**
     * Get email
     *
     * @return string
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * Set password
     *
     * @param string $password
     * @return User
     */
    public function setPassword($password)
    {
        $this->password = $password;

        return $this;
    }

    /**
     * Get password
     *
     * @return string
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * Set dateCreated
     *
     * @param \DateTime $dateCreated
     * @return User
     */
    public function setDateCreated($dateCreated)
    {
        $this->dateCreated = $dateCreated;

        return $this;
    }

    /**
     * Get dateCreated
     *
     * @return \DateTime
     */
    public function getDateCreated()
    {
        return $this->dateCreated;
    }

    /**
     * Set hashCode
     *
     * @param string $hashCode
     * @return User
     */
    public function setHashCode($hashCode)
    {
        $this->hashCode = $hashCode;

        return $this;
    }

    /**
     * Get hashCode
     *
     * @return string
     */
    public function getHashCode()
    {
        return $this->hashCode;
    }

    /**
     * Set status
     *
     * @param string $status
     * @return User
     */
    public function setStatus($status)
    {
        $this->status = $status;

        return $this;
    }

    /**
     * Get status
     *
     * @return string
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Set allowNewsletter
     *
     * @param boolean $allowNewsletter
     * @return User
     */
    public function setAllowNewsletter($allowNewsletter)
    {
        $this->allowNewsletter = $allowNewsletter;

        return $this;
    }

    /**
     * Get allowNewsletter
     *
     * @return boolean
     */
    public function getAllowNewsletter()
    {
        return $this->allowNewsletter;
    }

    /**
     * Add profile
     *
     * @param Profile $profile
     * @return User
     */
    public function addProfile(Profile $profile)
    {
        $this->profile[] = $profile;

        return $this;
    }

    /**
     * Remove profile
     *
     * @param Profile $profile
     */
    public function removeProfile(Profile $profile)
    {
        $this->profile->removeElement($profile);
    }

    /**
     * Get profile
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getProfile()
    {
        return $this->profile;
    }

    /**
     * Add billingData
     *
     * @param BillingData $billingData
     * @return User
     */
    public function addBillingData(BillingData $billingData)
    {
        $this->billingData[] = $billingData;

        return $this;
    }

    /**
     * Remove billingData
     *
     * @param BillingData $billingData
     */
    public function removeBillingData(BillingData $billingData)
    {
        $this->billingData->removeElement($billingData);
    }

    /**
     * Get billingData
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getBillingData()
    {
        return $this->billingData;
    }

    /**
     * Add userInterest
     *
     * @param UserInterest $userInterest
     * @return User
     */
    public function addUserInterest(UserInterest $userInterest)
    {
        $this->userInterests[] = $userInterest;

        return $this;
    }

    /**
     * Remove userInterest
     *
     * @param UserInterest $userInterest
     */
    public function removeUserInterest(UserInterest $userInterest)
    {
        $this->userInterests->removeElement($userInterest);
    }

    /**
     * Get userInterests
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getUserInterests()
    {
        return $this->userInterests;
    }

    /**
     * Add userAvatar
     *
     * @param UserAvatar $userAvatar
     * @return User
     */
    public function addUserAvatar(UserAvatar $userAvatar)
    {
        $this->userAvatars[] = $userAvatar;

        return $this;
    }

    /**
     * Remove userAvatar
     *
     * @param UserAvatar $userAvatar
     */
    public function removeUserAvatar(UserAvatar $userAvatar)
    {
        $this->userAvatars->removeElement($userAvatar);
    }

    /**
     * Get userAvatars
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getUserAvatars()
    {
        return $this->userAvatars;
    }

    /**
     * Add log
     *
     * @param Log $log
     * @return User
     */
    public function addLog(Log $log)
    {
        $this->logs[] = $log;

        return $this;
    }

    /**
     * Remove log
     *
     * @param Log $log
     */
    public function removeLog(Log $log)
    {
        $this->logs->removeElement($log);
    }

    /**
     * Get logs
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getLogs()
    {
        return $this->logs;
    }

    /**
     * Ustawia czy zaakceptował regulamin.
     *
     * @param bool $confirm
     * @return User
     */
    public function setConfirm($confirm)
    {
        $this->confirm = $confirm;

        return $this;
    }

    /**
     * Pobiera informację czy zaakceptował regulamin.
     *
     * @return bool
     */
    public function getConfirm()
    {
        return $this->confirm;
    }

    /**
     * Add favoriteNews
     *
     * @param News $favoriteNews
     * @return User
     */
    public function addFavoriteNews(News $favoriteNews)
    {
        $favoriteNews->addUser($this);
        $this->favoriteNews[] = $favoriteNews;

        return $this;
    }

    /**
     * Remove favoriteNews
     *
     * @param News $favoriteNews
     */
    public function removeFavoriteNews(News $favoriteNews)
    {
        $this->favoriteNews->removeElement($favoriteNews);
    }

    /**
     * Get favoriteNews
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getFavoriteNews()
    {
        return $this->favoriteNews;
    }

    /**
     * Set image
     *
     * @param string $image
     * @return User
     */
    public function setImage($image)
    {
        $this->image = $image;

        return $this;
    }

    /**
     * Get image
     *
     * @return string
     */
    public function getImage()
    {
        return $this->image;
    }

    // ------------------------------------------------------------
    // Methods.
    // ------------------------------------------------------------

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->billingData = new ArrayCollection();
        $this->logs = new ArrayCollection();
        $this->profile = new ArrayCollection();
        $this->userAvatars = new ArrayCollection();
        $this->userInterests = new ArrayCollection();
        $this->favoriteNews = new ArrayCollection();
    }

    /**
     * Zwraca nazwę użytkownika. Jeśli jest uzupełniony profil jest to imię,
     * w przeciwnym razie fragment adresu e-mail do małpy.
     *
     * @return string
     */
    public function getNickname() : string
    {
        if ($profile = $this->profile->first()) {
            if ($profile->getFirstName()) {
                return $profile->getFirstName();
            }
        }

        return preg_replace('/(@.*)/', '', $this->email);
    }

    /**
     * Zwraca link do avataru użytkownika.
     *
     * @return string
     */
    public function getAvatarLink() : string
    {
        if ($avatar = $this->userAvatars->first()) {
            return $avatar->getAvatar()->getImageLink();
        }

        return '';
    }

    /**
     * Pobiera zdjecie użytkownika lub avatar, jesli zdjęcie nie zostało
     * przesłane. Gdy nie ma przesłanego zdjęcia i nie ma wybranego awataru
     * to zwracany jest 'null'.
     *
     * @return string|null
     */
    public function getAvatar()
    {
        if (! empty($this->image)) {
            return 'media/assets/User/'.$this->image;
        }

        if ($this->userAvatars->count() > 0) {
            return 'media/avatars/' . $this->userAvatars->first()->getAvatar()->getImage();
        }

        return null;
    }
}
