<?php

namespace Wi\Admin\PageBundle\Repository;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\EntityRepository;

/**
 * Page Group Repository
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class PageGroupRepository extends EntityRepository
{
    /**
     * Liczba wszystkich aktywnych grup.
     *
     * @return int
     */
    public function countActive()
    {
        $qb = $this->createQueryBuilder('p');

        return (int) $qb
            ->select($qb->expr()->count('p'))
            ->where('p.status = :status')
            ->setParameters([
                'status' => 'A',
            ])
            ->getQuery()
            ->getSingleScalarResult()
        ;
    }

    /**
     * Pobiera wszystkie główne, aktywne grupy stron oraz należące do nich
     * aktywne strony. Sortowanie według kolejności i nazwy.
     *
     * @param string|null $search Tekst do wyszukania.
     * @return ArrayCollection
     */
    public function findAllActiveWithoutParent($search = '') : ArrayCollection
    {
        // Utworzenie QueryBuilder'a.
        $qb = $this->createQueryBuilder('pg')
            ->addSelect('p')
            ->leftJoin('pg.pages', 'p', 'WITH', 'p.status = :status')
            ->where('pg.status = :status')
            ->andWhere('pg.parent IS NULL')
        ;

        // Ustawienie parametrów dla zapytania.
        $params = [
            'status' => 'A',
        ];

        // Jeśli tekst do wyszukania jest przekazany to szukaj.
        if ($search) {
            $params['search'] = '%' . $search . '%';
            $qb->andWhere($qb->expr()->orX(
                $qb->expr()->like('p.title', ':search'),
                $qb->expr()->like('p.content', ':search'),
                $qb->expr()->like('p.metaTitle', ':search'),
                $qb->expr()->like('p.metaDescription', ':search'),
                $qb->expr()->like('p.metaKeywords', ':search'),
                $qb->expr()->like('p.metaCanonical', ':search'),
                $qb->expr()->like('p.metaImage', ':search'),
                $qb->expr()->like('p.visitCounter', ':search')
            ));
        }

        return new ArrayCollection(
            $qb
                ->setParameters($params)
                ->orderBy('pg.orderNr', 'ASC')
                ->addOrderBy('pg.name', 'ASC')
                ->getQuery()
                ->getResult()
        );
    }

    /**
     * Pobiera grupy stron wraz ze stronami dla generowania pliku sitemap.
     *
     * @return array
     */
    public function findForSitemap($ids = [])
    {
        return $this->createQueryBuilder('pg')
            ->addSelect('p')
            ->innerJoin('pg.pages', 'p')
            ->where('pg.id <> :group_id')
            ->orWhere('p.id IN (:ids)')
            ->andWhere('p.status = :status')
            ->setParameters([
                'group_id' => 1,
                'ids' => $ids,
                'status' => 'A',
            ])
            ->getQuery()
            ->getResult()
        ;
    }
}
