<?php

namespace Wi\Admin\PageBundle\Form;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Wi\Admin\PageBundle\Entity\PageGroup;

/**
 * Page Group Type.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class PageGroupType extends AbstractType
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * Construct.
     *
     * @param EntityManagerInterface $em
     */
    public function __construct(EntityManagerInterface $em)
    {
        $this->em = $em;
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $builder
            ->add('name', TextType::class, [
                'attr' => [
                    'maxlength' => 45,
                    'placeholder' => 'groupForm.name'
                ],
                'label' => 'groupForm.name',
            ])
            ->add('orderNr', null, [
                'attr' => [
                    'max' => 2147483646,
                    'maxlength' => 11,
                    'min' => -2147483646,
                ],
                'label' => 'groupForm.orderNr',
            ])
            ->add('parent', ChoiceType::class, [
                'attr' => [
                    'class' => 'chosenSelectGroup',
                ],
                'choice_attr' => function ($value, $key, $index) use ($options) {
                    if ($options['data']->getId() == $value->getId()) {
                        return ['disabled' => true];
                    }
                    return [];
                },
                'label_attr' => [
                    'class' => '',
                ],
                'choice_label' => function ($value, $key, $index) {
                    return $value->label;
                },
                'choice_translation_domain' => false,
                'choices' => $this->getChoices(),
                'label' => 'groupForm.parent',
                'placeholder' => 'groupForm.parentPlace',
                'required' => false,
            ])
        ;
    }

    /**
     * {@inheritdoc}
     */
    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults(array(
            'data_class' => PageGroup::class,
        ));
    }

    /**
     * {@inheritdoc}
     */
    public function getBlockPrefix()
    {
        return 'wi_admin_pagebundle_pagegroup';
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Przetwarza grupy stron i generuje z nich drzewo.
     *
     * @return PageGroup[]
     */
    private function getChoices()
    {
        $choices = [];

        // Pobranie wszystkich głównych, aktywnych grup stron.
        $pageGroups = $this->em->getRepository('WiAdminPageBundle:PageGroup')
            ->findAllActiveWithoutParent()
        ;

        /**
         * Przetwarza drzewo zagnieżdżeń kategorii i wypisuje do selecta.
         *
         * @param   PageGroup[] $pageGroups
         * @param   string      $prefix
         */
        $createTree = function ($pageGroups, $prefix = '') use (&$createTree, &$choices) {
            foreach ($pageGroups as $pageGroup) {
                // Pole label jest tworzone dynamicznie tylko na potrzeby wypisania etykiety.
                $pageGroup->label = $prefix . $pageGroup->getName();
                $choices[] = $pageGroup;

                if ($pageGroup->getChildren()->count()) {
                    $createTree($pageGroup->getChildren(), $prefix . '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;');
                }
            }

        };

        $createTree($pageGroups);

        return $choices;
    }
}
