<?php

namespace Wi\Admin\PageBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\PageBundle\Entity\Page;
use Wi\Admin\PageBundle\Form\PageDeleteType;
use Wi\Admin\PageBundle\Form\PageType;

/**
 * Page controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class PageController extends Controller
{
    /**
     * Lists all page entities.
     *
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie grup stron.
        $pageGroups = $em->getRepository('WiAdminPageBundle:PageGroup')
            ->findAllActiveWithoutParent($request->query->get('q'))
        ;

        return $this->render('WiAdminPageBundle:Page:index.html.php', [
            'pageGroups' => $pageGroups,
        ]);
    }

    /**
     * Creates a new page entity.
     *
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Utworzenie nowego obiektu strony.
        $page = new Page();

        // Dodanie domyślnej kategorii, jeśli została przekazana oraz istnieje.
        if ($request->query->getInt('page_group_id')) {
            $pageGroup = $em->getRepository('WiAdminPageBundle:PageGroup')
                ->findOneById($request->query->getInt('page_group_id'))
            ;

            if (! is_null($pageGroup)) {
                $page->setGroup($pageGroup);
            }
        }

        // Generowanie formularza.
        $form = $this->createForm(PageType::class, $page);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Zapis.
            $em->persist($page);
            $em->flush();

            return $this->redirectToRoute('wi_admin_pages_show', [
                'id' => $page->getId(),
            ]);
        }

        return $this->render('WiAdminPageBundle:Page:new.html.php', [
            'page' => $page,
            'form' => $form->createView(),
        ]);
    }

    /**
     * Finds and displays a page entity.
     *
     * @param   Page $page
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(Page $page)
    {
        return $this->render('WiAdminPageBundle:Page:show.html.php', [
            'page' => $page,
        ]);
    }

    /**
     * Displays a form to edit an existing page entity.
     *
     * @param   Request $request
     * @param   Page $page
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Page $page)
    {
        // Generowanie formularza.
        $editForm = $this->createForm(PageType::class, $page);
        $editForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($editForm->isSubmitted() && $editForm->isValid()) {
            // Zapis.
            $this->getDoctrine()->getManager()->flush();

            return $this->redirectToRoute('wi_admin_pages_show', [
                'id' => $page->getId(),
            ]);
        }

        return $this->render('WiAdminPageBundle:Page:edit.html.php', [
            'edit_form' => $editForm->createView(),
            'page' => $page,
        ]);
    }

    /**
     * Deletes a page entity.
     *
     * @param   Request $request
     * @param   Page $page
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, Page $page)
    {
        // Generowanie formularza.
        $deleteForm = $this->createForm(PageDeleteType::class);
        $deleteForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Usunięcie.
            $em->remove($page);
            $em->flush();

            return $this->redirectToRoute('wi_admin_pages_index');
        }

        return $this->render('WiAdminPageBundle:Page:delete.html.php', [
            'deleteForm' => $deleteForm->createView(),
            'page' => $page,
        ]);
    }

    /**
     * Obsługa edycji na żywo treści stron.
     *
     * @param Request $request
     * @param Page $page
     * @return JsonResponse
     */
    public function liveEditAction(Request $request, Page $page)
    {
        $pageJson = $request->request->get('page');

        if (empty($pageJson)) {
            return new JsonResponse([
                'message' => 'Empty params',
                'status' => false,
            ]);
        }

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Aktualizacja zawartości strony.
        if (! empty($content = $pageJson['content'])) {
            $page->setContent(json_decode($content));
        }

        $em->persist($page);
        $em->flush($page);

        return new JsonResponse([
            'status' => true,
        ]);
    }
}
