<?php

namespace Wi\Admin\NewsLetterBundle\Service;

use Doctrine\ORM\EntityManagerInterface;
use Wi\Admin\NewsLetterBundle\Entity\Recipients;
use Wi\Admin\NewsLetterBundle\Service\HashRecipientGroup;
use Wi\Admin\NewsLetterBundle\Service\RecipientMailer;

/**
 * Recipient manager.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2018 WEBimpuls Sp. z o.o.
 */
class RecipientManager
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * @var HashRecipientGroup
     */
    private $hasher;

    /**
     * @var RecipientMailer
     */
    private $recipientMailer;

    /**
     * @var Recipients
     */
    private $recipient;

    /**
     * Constructor.
     *
     * @param EntityManagerInterface $em
     * @param HashRecipientGroup $hasher
     * @param RecipientMailer $recipientMailer
     */
    public function __construct(EntityManagerInterface $em, HashRecipientGroup $hasher, RecipientMailer $recipientMailer)
    {
        $this->em = $em;
        $this->hasher = $hasher;
        $this->recipientMailer = $recipientMailer;
    }

    // -------------------------------------------------------------------------

    /**
     *
     *
     *
     *
     */
    public function signUpRecipient(array $params) : bool
    {
        $email = $params['email'];
        $name = $params['name'];
        $group = $this->getRecipientGroup($params['group']);

        if (! $group) { // Jeśli grupa nie istnieje.
            return false;
        }

        $recipient = new Recipients();
        $recipient
            ->setRecipientGroup($group)
            ->setEmail($email)
            ->setName($name)
        ;

        $this->em->persist($recipient);
        $this->em->flush($recipient);

        $this->recipient = $recipient;

        return true;
    }

    public function moveToNewsletter(array $params) : bool
    {
        $email = $params['email'];
        $name = $params['name'];
        $group = $this->em->getRepository('WiAdminNewsLetterBundle:RecipientGroup')
            ->find($params['group'])
        ;

        if (! $group) { // Jeśli grupa nie istnieje.
            return false;
        }

        $recipient = new Recipients();
        $recipient
            ->setRecipientGroup($group)
            ->setEmail($email)
            ->setName($name)
        ;

        $this->em->persist($recipient);
        $this->em->flush($recipient);

        $this->recipient = $recipient;

        return true;
    }

    public function generateActiveEmail()
    {
        $this->recipient
            ->setConsent(false)
            ->setHash(md5(uniqid(microtime() . $this->recipient->getEmail()) . $this->recipient->getId()))
        ;

        $this->em->persist($this->recipient);
        $this->em->flush($this->recipient);

        $result = $this->recipientMailer->sendConfirmEmail($this->recipient);
    }

    public function checkExist(array $params) : bool
    {
        $group = $this->getRecipientGroup($params['group']);
        $email = $params['email'];

        if ($recipient = $this->em->getRepository('WiAdminNewsLetterBundle:Recipients')->findOneBy([
            'recipientGroup' => $group,
            'email' => $email,
        ])) {
            $this->recipient = $recipient;

            return true;
        }

        return false;
    }

    /**
     *
     *
     *
     */
    public function getRecipient() : Recipients
    {
        return $this->recipient;
    }

    // -------------------------------------------------------------------------

    /**
     *
     *
     *
     *
     */
    private function getRecipientGroup(int $id)
    {
        return $this->em
            ->getRepository('WiAdminNewsLetterBundle:RecipientGroup')
            ->find($id)
        ;
    }
}
