<?php

namespace Wi\Admin\NewsLetterBundle\Repository;

use Doctrine\ORM\EntityRepository;
use Wi\Admin\NewsLetterBundle\Entity\Campaign;
use Wi\Admin\NewsLetterBundle\Entity\CampaignConfig;

/**
 * CampaignConfigRepository
 *
 * This class was generated by the Doctrine ORM. Add your own custom
 * repository methods below.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class CampaignConfigRepository extends EntityRepository
{
    const PARAM_BCC_NAME = 'bcc';
    const PARAM_DATE_END_NAME = 'date_end';
    const PARAM_DATE_START_NAME = 'date_start';
    const PARAM_DAYS_OF_WEEK_NAME = 'days_of_week';
    const PARAM_ENABLE_CAMPAIGN_NAME = 'enable_campaign';
    const PARAM_TIME_END_NAME = 'time_end';
    const PARAM_TIME_START_NAME = 'time_start';
    const PARAM_QUANTITY_TO_SENT_NAME = 'quantity_to_sent';
    const PARAM_RANDOMIZE_MESSAGES_NAME = 'randomize_messages';
    const PARAM_RANDOMIZE_RECIPIENTS_NAME = 'randomize_recipients';
    const PARAM_RANDOMIZE_SENDERS_NAME = 'randomize_senders';

    /**
     * Pobiera ukrytą kopię do wiadomości dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getBcc(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_BCC_NAME);
    }

    /**
     * Pobiera ukrytą kopię do wiadomości dla podanej kampanii lub tworzy nową.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getBccOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getBcc($campaign))) {
            return $config;
        }

        return $this->createConfig($campaign, self::PARAM_BCC_NAME);
    }

    /**
     * Pobiera aktualne ustawienie o podanym parametrze dla kampanii.
     *
     * @param Campaign $campaign
     * @param string $paramName
     * @return CampaignConfig|null
     */
    public function getCurrentConfigByParamName(Campaign $campaign, $paramName)
    {
        return $this->createQueryBuilder('c')
            ->where('c.paramName = :paramName')
            ->andWhere('c.campaign = :campaign')
            ->andWhere('c.dateDeleted IS NULL')
            ->setParameters([
                'campaign' => (! is_null($campaign->getId()) ? $campaign : null),
                'paramName' => $paramName,
            ])
            ->getQuery()
            ->getOneOrNullResult()
        ;
    }

    /**
     * Pobiera stan (ON/OFF) dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getEnableCampaign(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_ENABLE_CAMPAIGN_NAME);
    }

    /**
     * Pobiera stan (ON/OFF) dla podanej kampanii lub tworzy nowy.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getEnableCampaignOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getEnableCampaign($campaign))) {
            return $config;
        }

        return $this->createConfig($campaign, self::PARAM_ENABLE_CAMPAIGN_NAME, 0);
    }

    /**
     * Pobiera datę zakończenia dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getDateEnd(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_DATE_END_NAME);
    }

    /**
     * Pobiera datę zakończenia dla podanej kampanii lub tworzy nową.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getDateEndOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getDateEnd($campaign))) {
            return $config;
        }

        return $this->createConfig(
            $campaign,
            self::PARAM_DATE_END_NAME,
            (new \DateTime())->modify('+1 month')->format('Y-m-d')
        );
    }

    /**
     * Pobiera datę rozpoczęcia dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getDateStart(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_DATE_START_NAME);
    }

    /**
     * Pobiera datę rozpoczęcia dla podanej kampanii lub tworzy nową.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getDateStartOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getDateStart($campaign))) {
            return $config;
        }

        return $this->createConfig(
            $campaign,
            self::PARAM_DATE_START_NAME,
            (new \DateTime())->format('Y-m-d')
        );
    }

    /**
     * Pobiera dni tygodnia dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getDaysOfWeek(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_DAYS_OF_WEEK_NAME);
    }

    /**
     * Pobiera dni tygodnia dla podanej kampanii lub tworzy nowe.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getDaysOfWeekOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getDaysOfWeek($campaign))) {
            return $config;
        }

        return $this->createConfig($campaign, self::PARAM_DAYS_OF_WEEK_NAME, '1,2,3,4,5,6,7');
    }

    /**
     * Pobiera ilość wiadomości do wysłania dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getQuantityToSent(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_QUANTITY_TO_SENT_NAME);
    }

    /**
     * Pobiera ilość wiadomości do wysłania dla podanej kampanii lub tworzy nową.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getQuantityToSentOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getQuantityToSent($campaign))) {
            return $config;
        }

        return $this->createConfig($campaign, self::PARAM_QUANTITY_TO_SENT_NAME, 0);
    }

    /**
     * Pobiera losowość wiadomości dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getRandomizeMessages(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_RANDOMIZE_MESSAGES_NAME);
    }

    /**
     * Pobiera losowość wiadomości dla podanej kampanii lub tworzy nową.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getRandomizeMessagesOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getRandomizeMessages($campaign))) {
            return $config;
        }

        return $this->createConfig($campaign, self::PARAM_RANDOMIZE_MESSAGES_NAME, 0);
    }

    /**
     * Pobiera losowość odbiorców dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getRandomizeRecipients(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_RANDOMIZE_RECIPIENTS_NAME);
    }

    /**
     * Pobiera losowość odbiorców dla podanej kampanii lub tworzy nową.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getRandomizeRecipientsOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getRandomizeRecipients($campaign))) {
            return $config;
        }

        return $this->createConfig($campaign, self::PARAM_RANDOMIZE_RECIPIENTS_NAME, 0);
    }

    /**
     * Pobiera losowość nadawców dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getRandomizeSenders(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_RANDOMIZE_SENDERS_NAME);
    }

    /**
     * Pobiera losowość nadawców dla podanej kampanii lub tworzy nową.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getRandomizeSendersOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getRandomizeSenders($campaign))) {
            return $config;
        }

        return $this->createConfig($campaign, self::PARAM_RANDOMIZE_SENDERS_NAME, 0);
    }

    /**
     * Pobiera czas zakończenia dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getTimeEnd(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_TIME_END_NAME);
    }

    /**
     * Pobiera czas zakończenia dla podanej kampanii lub tworzy nowy.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getTimeEndOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getTimeEnd($campaign))) {
            return $config;
        }

        return $this->createConfig($campaign, self::PARAM_TIME_END_NAME, '16:00');
    }

    /**
     * Pobiera czas rozpoczęcia dla podanej kampanii.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getTimeStart(Campaign $campaign)
    {
        return $this->getCurrentConfigByParamName($campaign, self::PARAM_TIME_START_NAME);
    }

    /**
     * Pobiera czas rozpoczęcia dla podanej kampanii lub tworzy nowy.
     *
     * @param Campaign $campaign
     * @return CampaignConfig
     */
    public function getTimeStartOrCreate(Campaign $campaign)
    {
        if (! is_null($config = $this->getTimeStart($campaign))) {
            return $config;
        }

        return $this->createConfig($campaign, self::PARAM_TIME_START_NAME, '08:00');
    }

    // -------------------------------------------------------------------------

    /**
     * Pobiera informację, czy kampania jest aktywna lub nie.
     *
     * @param Campaign $campaign
     * @return bool
     */
    public function isEnabledCampaign(Campaign $campaign)
    {
        return (
            is_null($result = $this->getEnableCampaign($campaign))
            ? false
            : boolval(intval($result->getParamValue()))
        );
    }

    /**
     * Pobiera konfigurację kampanii.
     *
     * @param Campaign $campaign
     * @return array
     */
    public function getConfig(Campaign $campaign)
    {
        return [
            'enable_campaign' => $this->isEnabledCampaign($campaign),
            'quantity_to_sent' => (int) $this->getQuantityToSentOrCreate($campaign)->getParamValue(),
            'date_start' => ($this->getDateStartOrCreate($campaign)->getParamValue() <= date('Y-m-d')),
            'date_end' => ($this->getDateEndOrCreate($campaign)->getParamValue() >= date('Y-m-d')),
            'days_of_week' => in_array(date('N'), explode(',', $this->getDaysOfWeekOrCreate($campaign)->getParamValue())),
            'time_start' => (date('H:i') >= $this->getTimeStartOrCreate($campaign)->getParamValue()),
            'time_end' => (date('H:i') <= $this->getTimeEndOrCreate($campaign)->getParamValue()),
            'randomize_messages' => (bool) $this->getRandomizeMessagesOrCreate($campaign)->getParamValue(),
            'randomize_recipients' => (bool) $this->getRandomizeRecipientsOrCreate($campaign)->getParamValue(),
            'randomize_senders' => (bool) $this->getRandomizeSendersOrCreate($campaign)->getParamValue(),
            'bcc' => $this->getBccOrCreate($campaign)->getParamValue(),
        ];
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Tworzy nowy obiekt konfiguracji.
     *
     * @param Campaign $campaign
     * @param string $paramName
     * @param string $paramValue
     * @return CampaignConfig
     */
    private function createConfig(Campaign $campaign, $paramName, $paramValue = null)
    {
        return (new CampaignConfig())
            ->setCampaign($campaign)
            ->setParamName($paramName)
            ->setParamValue($paramValue)
        ;
    }
}
