<?php

namespace Wi\Admin\NewsLetterBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Symfony\Component\Validator\Constraints as Assert;
use Wi\Admin\NewsLetterBundle\Entity\Logs;
use Wi\Admin\NewsLetterBundle\Entity\MessageContent;
use Wi\Admin\NewsLetterBundle\Entity\CampaignConfig;
use Wi\Admin\NewsLetterBundle\Entity\Senders;
use Wi\Admin\NewsLetterBundle\Entity\RecipientGroup;

/**
 * Campaign
 *
 * @author Maksymilian Baryczka <mmaxioo33@gmail.com>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 *
 * @ORM\Table(name="newsletter_campaign")
 * @ORM\Entity(repositoryClass="Wi\Admin\NewsLetterBundle\Repository\CampaignRepository")
 * @ORM\EntityListeners({"Wi\Admin\NewsLetterBundle\EventListener\Entity\CampaignListener"})
 * @ORM\HasLifecycleCallbacks()
 * @Gedmo\SoftDeleteable(fieldName = "dateDeleted")
 */
class Campaign
{
    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(name="name", type="string", length=128, nullable=true)
     * @Assert\NotBlank()
     */
    private $name;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="date_expired", type="date", nullable=true)
     * @Assert\NotBlank()
     */
    private $dateExpired;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="date_created", type="datetime")
     */
    private $dateCreated;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="date_deleted", type="datetime", nullable=true)
     */
    private $dateDeleted;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\NewsLetterBundle\Entity\Logs", mappedBy="campaign")
     */
    private $logs;

    /**
     * @var string
     *
     * @ORM\Column(name="status", type="string", length=1, options={"default": "A", "fixed" = true})
     */
    private $status;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\NewsLetterBundle\Entity\MessageContent", mappedBy="campaign")
     */
    private $messageContent;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\NewsLetterBundle\Entity\CampaignConfig", mappedBy="campaign")
     */
    private $config;

    /**
     * @var ArrayCollection
     *
     * @ORM\ManyToMany(targetEntity="Wi\Admin\NewsLetterBundle\Entity\Senders", inversedBy="campaigns")
     * @ORM\JoinTable(name="newsletter_campaignSenderBind",
     *      joinColumns={@ORM\JoinColumn(name="campaign_id", referencedColumnName="id")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="sender_id", referencedColumnName="id")}
     *      )
     */
    private $senders;

    /**
     * @var ArrayCollection
     *
     * @ORM\ManyToMany(targetEntity="Wi\Admin\NewsLetterBundle\Entity\RecipientGroup")
     * @ORM\JoinTable(name="newsletter_campaignRecipientGroupBind",
     *      joinColumns={@ORM\JoinColumn(name="campaign_id", referencedColumnName="id")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="recipientGroup_id", referencedColumnName="id")}
     *      )
     */
    private $recipientGroups;


    // ------------------------------------------------------------
    // Has Lifecycle Callbacks methods.
    // ------------------------------------------------------------

    /**
     * Funkcja wykonuje się podczas wstawiania nowego rekordu do bazy.
     *
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        // Ustawienie daty utworzenia.
        $this->setDateCreated(new \DateTime());

        // Ustawienie statusu.
        $this->setStatus('A');
    }

     // ------------------------------------------------------------
    // Getters and setters methods.
    // ------------------------------------------------------------


    /**
     * Get id
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set name
     *
     * @param string $name
     *
     * @return Campaign
     */
    public function setName($name)
    {
        $this->name = $name;

        return $this;
    }

    /**
     * Get name
     *
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Set dateExpired
     *
     * @param \DateTime $dateExpired
     *
     * @return Campaign
     */
    public function setDateExpired($dateExpired)
    {
        $this->dateExpired = $dateExpired;

        return $this;
    }

    /**
     * Get dateExpired
     *
     * @return \DateTime
     */
    public function getDateExpired()
    {
        return $this->dateExpired;
    }

    /**
     * Set dateCreated
     *
     * @param \DateTime $dateCreated
     *
     * @return Campaign
     */
    public function setDateCreated($dateCreated)
    {
        $this->dateCreated = $dateCreated;

        return $this;
    }

    /**
     * Get dateCreated
     *
     * @return \DateTime
     */
    public function getDateCreated()
    {
        return $this->dateCreated;
    }

    /**
     * Set dateDeleted
     *
     * @param \DateTime $dateDeleted
     *
     * @return Campaign
     */
    public function setDateDeleted($dateDeleted)
    {
        $this->dateDeleted = $dateDeleted;

        return $this;
    }

    /**
     * Get dateDeleted
     *
     * @return \DateTime
     */
    public function getDateDeleted()
    {
        return $this->dateDeleted;
    }

    /**
     * Get logs
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getLogs()
    {
        return $this->logs;
    }

    /**
     * Set status
     *
     * @param string $status
     *
     * @return Campaign
     */
    public function setStatus($status)
    {
        $this->status = $status;

        return $this;
    }

    /**
     * Get status
     *
     * @return string
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Get messageContent
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getMessageContent()
    {
        return $this->messageContent;
    }

    /**
     * Add log
     *
     * @param Logs $log
     * @return Campaign
     */
    public function addLog(Logs $log)
    {
        $this->logs[] = $log;

        return $this;
    }

    /**
     * Remove log
     *
     * @param Logs $log
     */
    public function removeLog(Logs $log)
    {
        $this->logs->removeElement($log);
    }

    /**
     * Add messageContent
     *
     * @param MessageContent $messageContent
     * @return Campaign
     */
    public function addMessageContent(MessageContent $messageContent)
    {
        $this->messageContent[] = $messageContent;

        return $this;
    }

    /**
     * Remove messageContent
     *
     * @param MessageContent $messageContent
     */
    public function removeMessageContent(MessageContent $messageContent)
    {
        $this->messageContent->removeElement($messageContent);
    }

    /**
     * Add config
     *
     * @param CampaignConfig $config
     * @return Campaign
     */
    public function addConfig(CampaignConfig $config)
    {
        $config->setCampaign($this);
        $this->config[] = $config;

        return $this;
    }

    /**
     * Remove config
     *
     * @param CampaignConfig $config
     */
    public function removeConfig(CampaignConfig $config)
    {
        $this->config->removeElement($config);
    }

    /**
     * Get config
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getConfig()
    {
        return $this->config;
    }

    /**
     * Add sender
     *
     * @param Senders $sender
     * @return Campaign
     */
    public function addSender(Senders $sender)
    {
        $this->senders[] = $sender;

        return $this;
    }

    /**
     * Remove sender
     *
     * @param Senders $sender
     */
    public function removeSender(Senders $sender)
    {
        $this->senders->removeElement($sender);
    }

    /**
     * Get senders
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getSenders()
    {
        return $this->senders;
    }

    /**
     * Add recipientGroup
     *
     * @param RecipientGroup $recipientGroup
     * @return Campaign
     */
    public function addRecipientGroup(RecipientGroup $recipientGroup)
    {
        $this->recipientGroups[] = $recipientGroup;

        return $this;
    }

    /**
     * Remove recipientGroup
     *
     * @param RecipientGroup $recipientGroup
     */
    public function removeRecipientGroup(RecipientGroup $recipientGroup)
    {
        $this->recipientGroups->removeElement($recipientGroup);
    }

    /**
     * Get recipientGroups
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getRecipientGroups()
    {
        return $this->recipientGroups;
    }

    // ------------------------------------------------------------
    // Methods.
    // ------------------------------------------------------------

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->config = new ArrayCollection();
        $this->logs = new ArrayCollection();
        $this->messageContent = new ArrayCollection();
        $this->recipientGroups = new ArrayCollection();
        $this->senders = new ArrayCollection();
    }

    /**
     * Liczba wysłanych wiadomosci w kampani.
     *
     * @return int
     */
    public function getSendMessagesCount()
    {
        return count($this->logs->filter(function ($entry) {
            return $entry->getStatus() == 'A';
        }));
    }

    /**
     * Liczba wiadomości w kampanii.
     *
     * @return int
     */
    public function getMessagesCount()
    {
        return $this->messageContent->count();
    }

    public function getSendersCount()
    {
        return $this->senders->count();
    }

    public function getRecipientsCount()
    {
        $count = 0;

        foreach ($this->recipientGroups as $recipientGroup) {
            $count += $recipientGroup->getRecipientsCount();
        }

        return $count;
    }

    public function getAwaitingMessagesCount()
    {
        return $this->getRecipientsCount() - $this->getSendMessagesCount();
    }
}
