<?php

namespace Wi\Admin\NewsLetterBundle\Controller;

use Wi\Admin\NewsLetterBundle\Entity\Recipients;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\NewsLetterBundle\Form\RecipientsType;
use Wi\Admin\NewsLetterBundle\Form\RecipientsDeleteType;
use Symfony\Component\Serializer\Serializer;
use Symfony\Component\Serializer\Encoder\CsvEncoder;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Form\Extension\Core\Type\FileType;
use Symfony\Component\HttpFoundation\Response;

/**
 * Recipient controller.
 *
 * @author Maksymilian Baryczka <mmaxioo33@gmail.com>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class RecipientsController extends Controller
{
    /**
     * Lists all recipient entities.
     *
     */
    public function indexAction(Request $request)
    {
        $em = $this->getDoctrine()->getManager();

        $paginator = $this->get('knp_paginator');
        $recipients = $paginator->paginate(
            $em->getRepository('WiAdminNewsLetterBundle:Recipients')
                ->findAllActive(),
            $request->query->getInt('page', 1),
            $request->query->getInt('per_page') ?: 100
        );

        $form = $this->createFormBuilder()
            ->add('file', FileType::class, [
                'attr' => [
                    'accept' => '.csv',
                    'class' => 'InputFile',
                ],
                'label' => false,
                'mapped' => false,
                'required' => true,
            ])
            ->getForm();
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $file = $form->get('file')->getData();

            if (($handle = fopen($file, 'r')) !== false) {
                while (($row = fgetcsv($handle)) !== false) {
                    $email = ! empty($row[0]) ? $row[0] : '';
                    $name = ! empty($row[1]) ? $row[1] : null;
                    $lang = ! empty($row[2]) ? mb_strtolower($row[2]) : 'pl';
                    $consent = ! empty($row[3]) ? boolval(intval($row[3])) : false;
                    $notUnique = (bool) $em->getRepository('WiAdminNewsLetterBundle:Recipients')->findOneByEmail($email);

                    if (! $notUnique) {
                        $recipient = new Recipients();
                        $recipient
                            ->setEmail($email)
                            ->setName($name)
                            ->setLang($lang)
                            ->setConsent($consent)
                        ;
                        $em->persist($recipient);
                    }
                }

                $em->flush();
            }

            return $this->redirectToRoute('wi_admin_newsletter_recipients_index');
        }

        return $this->render('WiAdminNewsLetterBundle:Recipients:index.html.php', array(
            'recipients' => $recipients,
            'form' => $form->createView(),
        ));
    }

    /**
     * Creates a new recipient entity.
     *
     */
    public function newAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        $recipientGroup = $em->getRepository('WiAdminNewsLetterBundle:RecipientGroup')->find($request->query->getInt('rid'));

        // Utworzenie nowego obiektu pytań.
        $recipient = new Recipients();

        // Generowanie formularza.
        $form = $this->createForm(RecipientsType::class, $recipient);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            $recipient->setRecipientGroup($recipientGroup);
            // Zapis.
            $em->persist($recipient);
            $em->flush();

            return $this->redirectToRoute('wi_admin_newsletter_recipient_group_show', [
                'id' => $recipient->getRecipientGroup()->getId(),
            ]);
        }

        return $this->render('WiAdminNewsLetterBundle:Recipients:new.html.php', [
            'form' => $form->createView(),
            'recipient' => $recipient,
            'recipientGroup' => $recipientGroup,
        ]);
    }

    /**
     * Finds and displays a recipient entity.
     *
     */
    public function showAction(Recipients $recipients)
    {

        return $this->render('WiAdminNewsLetterBundle:Recipients:show.html.php', array(
            'recipients' => $recipients,
        ));
    }

    /**
     * Displays a form to edit an existing recipient entity.
     *
     */
    public function editAction(Request $request, Recipients $recipient)
    {
        // Generowanie formularza.
        $form = $this->createForm(RecipientsType::class, $recipient);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Zapis.
            $em->persist($recipient);
            $em->flush();

            return $this->redirectToRoute('wi_admin_newsletter_recipient_group_show', [
                'id' => $recipient->getRecipientGroup()->getId(),
            ]);
        }

        return $this->render('WiAdminNewsLetterBundle:Recipients:edit.html.php', [
            'form' => $form->createView(),
            'recipient' => $recipient,
        ]);
    }

    /**
     * Deletes a recipient entity.
     *
     */
    public function deleteAction(Request $request, Recipients $recipient)
    {
        // Generowanie formularza.
        $deleteForm = $this->createForm(RecipientsDeleteType::class);
        $deleteForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
            $id = $recipient->getRecipientGroup()->getId();

            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Usunięcie.
            $em->remove($recipient);
            $em->flush();

            return $this->redirectToRoute('wi_admin_newsletter_recipient_group_show', [
                'id' => $id,
            ]);
        }

        return $this->render('WiAdminNewsLetterBundle:Recipients:delete.html.php', [
            'deleteForm' => $deleteForm->createView(),
            'recipient' => $recipient,
        ]);
    }

     public function exportlogAction()
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $logs = $em->getRepository('WiAdminNewsLetterBundle:Logs')->findAll();

        $serializer = new Serializer([new ObjectNormalizer()], [new CsvEncoder()]);

        // instantiation, when using it inside the Symfony framework

        // encoding contents in CSV format
        $serializer->encode($logs, 'csv');

        file_put_contents(
            'data.csv',
            $container->get('serializer')->encode($logs, 'csv')
        );
    }

}
