<?php

namespace Wi\Admin\NewsLetterBundle\Controller;

use Knp\Component\Pager\Paginator;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\NewsLetterBundle\Entity\Campaign;
use Wi\Admin\NewsLetterBundle\Entity\CampaignConfig;
use Wi\Admin\NewsLetterBundle\Form\CampaignAddSenderType;
use Wi\Admin\NewsLetterBundle\Form\CampaignAddRecipientGroupType;
use Wi\Admin\NewsLetterBundle\Form\CampaignDeleteType;
use Wi\Admin\NewsLetterBundle\Form\CampaignType;

/**
 * Campaign controller.
 *
 * @author Maksymilian Baryczka <mmaxioo33@gmail.com>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class CampaignController extends Controller
{
    /**
     * Widok podsumowania ustawień kampanii.
     *
     * @param Request $request
     * @param Paginator $paginator
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction(Request $request, Paginator $paginator, Campaign $campaign)
    {
        $em = $this->getDoctrine()->getManager();
        $logRepo = $em->getRepository('WiAdminNewsLetterBundle:Logs');

        // Logi.
        $logs = $paginator->paginate(
            $logRepo->getLogsByCampaign($campaign->getId()),
            $request->query->getInt('page', 1),
            $request->query->getInt('per_page') ?: 10,
            [
                'defaultSortDirection' => 'DESC',
                'defaultSortFieldName' => 'l.dateMessage',
            ]
        );

        $addSenderForm = $this->createForm(CampaignAddSenderType::class, $campaign);
        $addSenderForm->handleRequest($request);

        if ($addSenderForm->isSubmitted() && $addSenderForm->isValid()) {
            $em->persist($campaign);
            $em->flush();

            return $this->redirectToRoute('wi_admin_newsletter_campaign_index', [
                'id' => $campaign->getId(),
            ]);
        }

        $addGroupForm = $this->createForm(CampaignAddRecipientGroupType::class, $campaign);
        $addGroupForm->handleRequest($request);

        if ($addGroupForm->isSubmitted() && $addGroupForm->isValid()) {
            $em->persist($campaign);
            $em->flush();

            return $this->redirectToRoute('wi_admin_newsletter_campaign_index', [
                'id' => $campaign->getId(),
            ]);
        }

        $messageSendedCount = [];

        foreach ($campaign->getMessageContent() as $mContent) {
            $messageSendedCount[$mContent->getId()] = $logRepo->countByMessageContent($mContent, $campaign);
        }

        $sendersSendedCount = [];

        foreach ($campaign->getSenders() as $sender) {
            $sendersSendedCount[$sender->getId()] = $logRepo->countBySender($sender, $campaign);
        }

        // Ustawienia.
        $config = $this->getConfig($campaign);
        $statistics = $this->getStatistics($campaign, $config);

        return $this->render('WiAdminNewsLetterBundle:Campaign:index.html.php', [
            'addGroupForm' => $addGroupForm->createView(),
            'addSenderForm' => $addSenderForm->createView(),
            'campaign' => $campaign,
            'config' => $config,
            'logs' => $logs,
            'messageSendedCount' => $messageSendedCount,
            'sendersSendedCount' => $sendersSendedCount,
            'statistics' => $statistics,
        ]);
    }

    /**
     * Creates a new campaign entity.
     *
     */
    public function newAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Utworzenie nowego obiektu pytań.
        $campaign = new Campaign();

        // Generowanie formularza.
        $form = $this->createForm(CampaignType::class, $campaign);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            foreach ($this->getConfig($campaign) as $config) {
                $em->persist($config);
            }

            // Zapis.
            $em->persist($campaign);
            $em->flush();

            return $this->redirectToRoute('wi_admin_newsletter_campaign_index', [
                'id' => $campaign->getId(),
            ]);
        }

        return $this->render('WiAdminNewsLetterBundle:Campaign:new.html.php', [
            'form' => $form->createView(),
            'campaign' => $campaign,
        ]);
    }

    /**
     * Finds and displays a campaign entity.
     *
     */
    public function showAction(Campaign $campaign)
    {

        return $this->render('WiAdminNewsLetterBundle:Campaign:show.html.php', array(
            'campaign' => $campaign,
        ));
    }

    /**
     * Displays a form to edit an existing campaign entity.
     *
     */
    public function editAction(Request $request, Campaign $campaign)
    {
        // Generowanie formularza.
        $form = $this->createForm(CampaignType::class, $campaign);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Zapis.
            $em->persist($campaign);
            $em->flush();

            return $this->redirectToRoute('wi_admin_newsletter_mainsite_index', [
                'id' => $campaign->getId(),
            ]);
        }

        return $this->render('WiAdminNewsLetterBundle:Campaign:edit.html.php', [
            'form' => $form->createView(),
            'campaign' => $campaign,
        ]);
    }

    /**
     * Deletes a campaign entity.
     *
     */
    public function deleteAction(Request $request, Campaign $campaign)
    {
        // Generowanie formularza.
        $deleteForm = $this->createForm(CampaignDeleteType::class);
        $deleteForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Usunięcie.
            $em->remove($campaign);
            $em->flush();

            return $this->redirectToRoute('wi_admin_newsletter_mainsite_index');
        }

        return $this->render('WiAdminNewsLetterBundle:Campaign:delete.html.php', [
            'deleteForm' => $deleteForm->createView(),
            'campaign' => $campaign,
        ]);
    }

    /**
     * Zapis konfiguracji kampanii.
     *
     * @param Campaign $campaign
     * @return JsonResponse
     */
    public function ajaxSaveConfigAction(Request $request, Campaign $campaign)
    {
        $em = $this->getDoctrine()->getManager();
        $configRepo = $em->getRepository('WiAdminNewsLetterBundle:CampaignConfig');
        $id = $request->request->getInt('id');
        $paramName = $request->request->get('paramName');
        $paramValue = $request->request->get('paramValue') ?: null;
        $aidValue = $request->request->get('aidValue') ?: null;

        $config = $configRepo->find($id);
        $em->remove($config);

        switch ($paramName) {
            case 'days_of_week':
                $paramValue = rtrim($paramValue, ',');
                break;
            case 'enable_campaign':
                // no break
            case 'randomize_messages':
                // no break
            case 'randomize_recipients':
                // no break
            case 'randomize_senders':
                $paramValue = (int) $paramValue;
                break;
            default:
                break;
        }

        $config = new CampaignConfig();
        $config
            ->setCampaign($campaign)
            ->setParamName($paramName)
            ->setParamValue($paramValue)
            ->setAidValue($aidValue)
        ;

        $em->persist($config);
        $em->flush();

        return new JsonResponse([
            'status' => 'OK',
            'new_id' => $config->getId(),
        ]);
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Pobiera konfiguracje dla kampanii.
     *
     * @param Campaign $campaign
     * @return array
     */
    private function getConfig(Campaign $campaign)
    {
        $configRepo = $this->getDoctrine()->getManager()
            ->getRepository('WiAdminNewsLetterBundle:CampaignConfig')
        ;

        return [
            'enable_campaign' => $configRepo->getEnableCampaignOrCreate($campaign),
            'quantity_to_sent' => $configRepo->getQuantityToSentOrCreate($campaign),
            'date_start' => $configRepo->getDateStartOrCreate($campaign),
            'date_end' => $configRepo->getDateEndOrCreate($campaign),
            'days_of_week' => $configRepo->getDaysOfWeekOrCreate($campaign),
            'time_start' => $configRepo->getTimeStartOrCreate($campaign),
            'time_end' => $configRepo->getTimeEndOrCreate($campaign),
            'randomize_messages' => $configRepo->getRandomizeMessagesOrCreate($campaign),
            'randomize_recipients' => $configRepo->getRandomizeRecipientsOrCreate($campaign),
            'randomize_senders' => $configRepo->getRandomizeSendersOrCreate($campaign),
            'bcc' => $configRepo->getBccOrCreate($campaign),
        ];
    }

    private function getStatistics(Campaign $campaign, $config)
    {
        $statistics = [
            'campaign_active' => false,
            'date_range' => '',
            'content_messages' => $campaign->getMessagesCount(),
            'recipient_groups' => $campaign->getRecipientGroups()->count(),
            'recipients' => $campaign->getRecipientsCount(),
            'messages_sent' => $campaign->getSendMessagesCount(),
            'messages_awaiting' => $campaign->getAwaitingMessagesCount(),
            'senders' => $campaign->getSendersCount(),
            'info' => 'Campaign "'.$campaign->getName().'" is not active',
        ];

        if (boolval($config['enable_campaign']->getParamValue())) {
            $dateStart = (new \DateTime($config['date_start']->getParamValue()))->format('Y-m-d');
            $dateEnd = (new \DateTime($config['date_end']->getParamValue()))->format('Y-m-d');
            $now = (new \DateTime())->format('Y-m-d');
            $statistics['date_range'] = $dateStart.' - '.$dateEnd;

            if ($dateStart <= $now && $dateEnd >= $now) {
                if ($campaign->getDateExpired()->format('Y-m-d') >= $now) {
                    $statistics['campaign_active'] = true;
                    $statistics['info'] = '<span style="color: green;">Campaign "'.$campaign->getName().'" is active</span>';
                } else {
                    $statistics['info'] = 'Campaign "'.$campaign->getName().'" expired';
                }
            } else {
                $statistics['info'] = 'Date doesn\'t allow to run campaign "'.$campaign->getName().'" ';
            }
        }

        return $statistics;
    }
}
