<?php

namespace Wi\Admin\NewsLetterBundle\Command\Cron;

use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\ArrayInput;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Wi\Admin\NewsLetterBundle\Entity\Logs;

class NewslettersCommand extends ContainerAwareCommand
{
    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this
            ->setName('app:cron:newsletters')
            ->setDescription('Wysylanie wiadomosci do listy mailingowej.')
            ->setHelp('Wysylanie wiadomosci do listy mailingowej.')
            ->addOption(
                'campaign',
                null,
                InputOption::VALUE_REQUIRED,
                'ID of campaign to run',
                null
            )
        ;
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $output->writeln('Wysylanie wiadomosci.');

        $em = $this->getContainer()->get('doctrine.orm.entity_manager');
        $configMain = $em->getRepository('WiAdminNewsLetterBundle:Config')->getConfig();

        // Sprawdzenie czy newsletter jest włączony.
        if (! $configMain['enable_newsletter']) {
            $output->writeln('Newsletter wylaczony.');
            return;
        }

        $output->writeln('Newsletter wlaczony.');
        $configRepo = $em->getRepository('WiAdminNewsLetterBundle:CampaignConfig');

        if (! is_null($cid = $input->getOption('campaign'))) {
            $campaign = $em->getRepository('WiAdminNewsLetterBundle:Campaign')->find($cid);
            $campaigns = [];
            $campaigns[] = $campaign;
        } else {
            $campaigns = $em->getRepository('WiAdminNewsLetterBundle:Campaign')->getFindAllActive();
        }

        $emailCount = 0;

        foreach ($campaigns as $campaign) {
            $config = $configRepo->getConfig($campaign);

            // Sprawdzenie, czy data wysłania newslettera nadeszła.
            if (! $config['date_start'] || ! $config['date_end'] || ! $config['days_of_week'] || ! $config['time_start'] || ! $config['time_end']) {
                $output->writeln('Data nie pozwala na uruchomienie kampanii.');
                continue;
            }

            $output->writeln('Nadeszla data uruchomienia kampanii.');

            // Sprawdzenie, czy kampania jest ważna.
            if ($campaign->getDateExpired()->format('Y-m-d') < (new \DateTime())->format('Y-m-d')) {
                $output->writeln('Kampania stracila waznosc.');
                continue;
            }

            $output->writeln('Aktywna kampania: <info>'.$campaign->getName().'</info>.');

            // Sprawdzenie liczby wiadomości do wysłania jednorazowo.
            if ($config['quantity_to_sent'] <= 0) {
                $output->writeln('Liczba wiadomosci do wyslania jest ustawiona na 0.');
                continue;
            }

            $groups = [];

            foreach ($campaign->getRecipientGroups() as $recipientGroup) {
                $groups[] = $recipientGroup->getId();
            }

            // Pobranie odbiorców.
            $recipients = $em->getRepository('WiAdminNewsLetterBundle:Recipients')
                ->findRecipientsConsentedAndWasNoMessageSent(
                    $campaign,
                    $groups,
                    $config['quantity_to_sent'],
                    $config['randomize_recipients']
                )
            ;

            // Czy są wybrani odbiorcy.
            if (empty($recipients)) {
                $output->writeln('Liczba wybranych odbiorcow wynosi 0.');
                continue;
            }

            foreach ($recipients as $recipient) {
                $sender = $em->getRepository('WiAdminNewsLetterBundle:Senders')
                    ->findSenderByCampaign($campaign, $config['randomize_senders'])
                ;
                $note = '';

                if (! is_null($sender)) {
                    $message = $em->getRepository('WiAdminNewsLetterBundle:MessageContent')
                        ->findMessageContentByCampaing(
                            $campaign,
                            $config['randomize_messages']
                        )
                    ;

                    if (! is_null($message)) {
                        $to = $recipient->getEmail();

                        if (! empty($name = $recipient->getName())) {
                            $to = [$to => $name];
                        }

                        try {
                            $mailer = $this->getContainer()->get('wi.admin.newsletterbundle.mailer');
                            $mailer->init(['bcc' => $config['bcc'], 'sender' => $sender]);
                            $mailer->message
                                ->setTo($to)
                                ->setSubject($message->getTitle())
                                ->setBody($message->getMessage(), 'text/html')
                            ;
                            $emailCount += $mailer->send();
                        } catch (\Swift_TransportException $e) {
                            $note = $e->getMessage();
                        }
                    } else {
                        $note = 'Nie znaleziono wiadomosci dla kampanii: '.$campaign->getName().'.';
                        $output->writeln('Nie znaleziono wiadomosci dla kampanii: <info>'.$campaign->getName().'</info>.');
                    }
                } else {
                    $note = 'Nie znaleziono nadawcy dla kampanii: '.$campaign->getName().'.';
                    $output->writeln('Nie znaleziono nadawcy dla kampanii: <info>'.$campaign->getName().'</info>.');
                }

                $logs = new Logs();
                $logs
                    ->setRecipient($recipient)
                    ->setSender($sender)
                    ->setCampaign($campaign)
                    ->setMessageContent((! empty($message) ? $message : null))
                    ->setNote($note)
                ;

                if (! empty($note)) {
                    $logs->setStatus('E');
                }

                $em->persist($logs);
            }
        }

        $em->flush();
        $output->writeln('Wyslano: <info>'.$emailCount.'</info> wiadomosci.');
    }
}
