<?php

namespace Wi\Admin\NewsBundle\Repository;

use Doctrine\ORM\EntityRepository;
use Wi\Admin\NewsBundle\Entity\Comment;
use Wi\Admin\NewsBundle\Entity\News;

/**
 * CommentRepository
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2018 WEBimpuls Sp. z o.o.
 */
class CommentRepository extends EntityRepository
{
    /**
     * Pobiera określoną przez limit liczbę najnowszych komentarzy
     * dla określonej akutalności.
     *
     * @param News $news
     * @param int $limit Domyślnie 0 (0 oznacza brak limitu).
     * @return array
     */
    public function findWithLimit(News $news, int $limit = 0) : array
    {
        $qb = $this->createQueryBuilder('c');

        if ($limit > 0) {
            $qb->setMaxResults($limit);
        }

        return $qb
            ->where('c.news = :news')
            ->andWhere('c.parent IS NULL')
            ->andWhere('c.status = :status')
            ->orderBy('c.dateCreated', 'DESC')
            ->setParameters([
                'news' => $news,
                'status' => 'A',
            ])
            ->getQuery()
            ->getResult()
        ;
    }

    public function getAllQuery()
    {
        return $this->createQueryBuilder('c')
            ->addSelect('n, g, u')
            ->innerJoin('c.news', 'n')
            ->leftJoin('c.user', 'u')
            ->leftJoin('c.guest', 'g')
            ->where('c.status NOT IN (:statuses)')
            ->setParameters([
                'statuses' => ['S'],
            ])
            ->getQuery()
        ;
    }

    /**
     * Liczba wszystkich aktywnych komentarzy.
     *
     * @return int
     */
    public function countActive()
    {
        $qb = $this->createQueryBuilder('c');

        return (int) $qb
            ->select($qb->expr()->count('c'))
            ->where('c.status IN (:status)')
            ->setParameters([
                'status' => [Comment::STATUS_ACTIVE, Comment::STATUS_HIDDEN, Comment::STATUS_WAIT],
            ])
            ->getQuery()
            ->getSingleScalarResult()
            ;
    }

    /**
     * Liczba wszystkich wpisów powiązanych z komentarzami.
     *
     * @return int
     */
    public function countPostActive()
    {
        $qb = $this->createQueryBuilder('c');

        return (int) $qb
            ->select($qb->expr()->countDistinct('c.news'))
            ->where('c.status IN (:status)')
            ->setParameters([
                'status' => [
                    Comment::STATUS_ACTIVE,
                    Comment::STATUS_HIDDEN,
                    Comment::STATUS_WAIT
                ],
            ])
            ->getQuery()
            ->getSingleScalarResult()
        ;
    }
}
