<?php

namespace Wi\Admin\NewsBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;
use Wi\Admin\AdminBundle\Entity\Administrator;
use Wi\Admin\CoreBundle\Utils\Utils;
use Wi\Admin\DownloadBundle\Entity\File;
use Wi\Admin\NewsBundle\Entity\Category;
use Wi\Admin\NewsBundle\Entity\NewsImage;
use Wi\Admin\NewsBundle\Entity\Tag;
use Wi\Admin\UserBundle\Entity\User;

/**
 * News
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 *
 * @ORM\Table(name="news")
 * @ORM\HasLifecycleCallbacks()
 * @ORM\Entity(repositoryClass="Wi\Admin\NewsBundle\Repository\NewsRepository")
 * @ORM\EntityListeners({"Wi\Admin\NewsBundle\EventListener\Entity\NewsListener"})
 * @Gedmo\SoftDeleteable(fieldName = "dateDeleted")
 */
class News
{
    /**
     * Oznaczenie typu kategorii, jako news.
     *
     * @const TYPE_NEWS
     */
    const TYPE_NEWS = 'N';

    /**
     * Oznaczenie typu kategorii, jako blog.
     *
     * @const TYPE_BLOG
     */
    const TYPE_BLOG = 'B';

    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     * @Groups({"news_list", "news"})
     */
    private $id;

    /**
     * @var Wi\Admin\AdminBundle\Entity\Administrator
     *
     * @ORM\ManyToOne(targetEntity="Wi\Admin\AdminBundle\Entity\Administrator")
     * @ORM\JoinColumn(name="autor_id", referencedColumnName="id", nullable=false)
     */
    private $author;

    /**
     * @var string
     *
     * @ORM\Column(name="title", type="string", length=256)
     * @Assert\NotBlank()
     * @Groups({"news_list", "news"})
     */
    private $title;

    /**
     * @var string
     *
     * @ORM\Column(name="shortcut", type="string", length=512, nullable=true)
     * @Groups({"news_list", "news"})
     */
    private $shortcut;

    /**
     * @var string
     *
     * @ORM\Column(name="content", type="text", length=65535)
     * @Assert\NotBlank()
     * @Groups({"news_list", "news"})
     */
    private $content;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateOfPublication", type="datetime", nullable=true)
     * @Groups({"news_list", "news"})
     */
    private $dateOfPublication;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateCreated", type="datetime")
     */
    private $dateCreated;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateDeleted", type="datetime", nullable=true)
     */
    private $dateDeleted;

    /**
     * @var string
     *
     * @ORM\Column(name="metaTitle", type="string", length=256, nullable=true)
     */
    private $metaTitle;

    /**
     * @var string
     *
     * @ORM\Column(name="metaDescription", type="string", length=512, nullable=true)
     */
    private $metaDescription;

    /**
     * @var string
     *
     * @ORM\Column(name="metaKeywords", type="string", length=256, nullable=true)
     */
    private $metaKeywords;

    /**
     * @var string
     *
     * @ORM\Column(name="metaCanonical", type="string", length=512, nullable=true)
     */
    private $metaCanonical;

    /**
     * @var string
     *
     * @ORM\Column(name="metaImage", type="string", length=256, nullable=true)
     */
    private $metaImage;

    /**
     *
     * Dostępne statusy:
     * A - aktywny
     * N - nieopublikowany
     * D - usunięty
     *
     * @var string
     *
     * @ORM\Column(name="status", type="string", length=1, options={"default": "A", "fixed" = true})
     */
    private $status;

    /**
     * @var int
     *
     * @ORM\Column(name="visitCounter", type="integer", options={"default": "0"})
     */
    private $visitCounter;

    /**
     * @var string
     *
     * @ORM\Column(name="movie", type="string", length=255, nullable=true)
     */
    private $movie;

    /**
     * @var ArrayCollection
     *
     * @ORM\ManyToMany(targetEntity="Wi\Admin\NewsBundle\Entity\Category", inversedBy="news")
     * @ORM\JoinTable(name="news_categoriesBind",
     *      joinColumns={@ORM\JoinColumn(name="news_id", referencedColumnName="id")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="category_id", referencedColumnName="id")}
     * )
     * @ORM\OrderBy({"name" = "ASC"})
     * @Assert\Count(
     *      min = 1,
     * )
     * @Groups({"news_list", "news"})
     */
    private $categories;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\NewsBundle\Entity\NewsImage", mappedBy="news")
     * @ORM\OrderBy({"orderNr" = "ASC"})
     */
    private $images;

    /**
     * @var string
     *
     * @ORM\Column(name="type", type="string", length=1)
     * @Assert\NotBlank()
     */
    private $type;

    /**
     * @var boolean
     *
     * @ORM\Column(name="enableDownload", type="boolean")
     */
    private $enableDownload;

    /**
     * @var ArrayCollection
     *
     * @ORM\ManyToMany(targetEntity="Wi\Admin\NewsBundle\Entity\Tag")
     * @ORM\JoinTable(name="news_tagsBind",
     *     joinColumns={@ORM\JoinColumn(name="news_id", referencedColumnName="id")},
     *     inverseJoinColumns={@ORM\JoinColumn(name="tags_id", referencedColumnName="id")}
     *     )
     */
    private $tags;

    /**
     * @ORM\ManyToMany(targetEntity="\Wi\Admin\UserBundle\Entity\User", mappedBy="favoriteNews")
     */
    private $users;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\DownloadBundle\Entity\File", mappedBy="news")
     */
    private $files;


    // ------------------------------------------------------------
    // Has Lifecycle Callbacks methods.
    // ------------------------------------------------------------

    /**
     * Funkcja wykonuje się podczas wstawiania nowego rekordu do bazy.
     *
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        // Ustawienie daty utworzenia.
        $this->setDateCreated(new \DateTime());

        // Ustawienie statusu.
        $this->setStatus('A');

        // Ustawienie licznika odczytań aktualności na 0.
        $this->setVisitCounter(0);

        // Wstawienie krótkiego opisu na podstawie treści, jeśli krótki opis
        // pozostał pusty.
        $this->setShortcutWhenIsNull();
    }

    /**
     * Funkcja wykonuje się podczas aktualizacji rekordu z bazy.
     *
     * @ORM\PreUpdate
     */
    public function preUpdate()
    {
        // Wstawienie krótkiego opisu na podstawie treści, jeśli krótki opis
        // pozostał pusty.
        $this->setShortcutWhenIsNull();
    }

    // ------------------------------------------------------------
    // User Interface methods.
    // ------------------------------------------------------------

    /**
     * Get id
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set title
     *
     * @param string $title
     * @return News
     */
    public function setTitle($title)
    {
        $this->title = $title;

        return $this;
    }

    /**
     * Get title
     *
     * @return string
     */
    public function getTitle()
    {
        return $this->title;
    }

    /**
     * Set shortcut
     *
     * @param string $shortcut
     * @return News
     */
    public function setShortcut($shortcut)
    {
        $this->shortcut = $shortcut;

        return $this;
    }

    /**
     * Get shortcut
     *
     * @return string
     */
    public function getShortcut()
    {
        return $this->shortcut;
    }

    /**
     * Set content
     *
     * @param string $content
     * @return News
     */
    public function setContent($content)
    {
        $this->content = $content;

        return $this;
    }

    /**
     * Get content
     *
     * @return string
     */
    public function getContent()
    {
        return $this->content;
    }

    /**
     * Set dateOfPublication
     *
     * @param \DateTime $dateOfPublication
     * @return News
     */
    public function setDateOfPublication($dateOfPublication)
    {
        $this->dateOfPublication = $dateOfPublication;

        return $this;
    }

    /**
     * Get dateOfPublication
     *
     * @return \DateTime
     */
    public function getDateOfPublication()
    {
        return $this->dateOfPublication;
    }

    /**
     * Set dateCreated
     *
     * @param \DateTime $dateCreated
     * @return News
     */
    public function setDateCreated($dateCreated)
    {
        $this->dateCreated = $dateCreated;

        return $this;
    }

    /**
     * Get dateCreated
     *
     * @return \DateTime
     */
    public function getDateCreated()
    {
        return $this->dateCreated;
    }

    /**
     * Set dateDeleted
     *
     * @param \DateTime $dateDeleted
     * @return News
     */
    public function setDateDeleted($dateDeleted)
    {
        $this->dateDeleted = $dateDeleted;

        return $this;
    }

    /**
     * Get dateDeleted
     *
     * @return \DateTime
     */
    public function getDateDeleted()
    {
        return $this->dateDeleted;
    }

    /**
     * Set metaTitle
     *
     * @param string $metaTitle
     * @return News
     */
    public function setMetaTitle($metaTitle)
    {
        $this->metaTitle = $metaTitle;

        return $this;
    }

    /**
     * Get metaTitle
     *
     * @return string
     */
    public function getMetaTitle()
    {
        return $this->metaTitle;
    }

    /**
     * Set metaDescription
     *
     * @param string $metaDescription
     * @return News
     */
    public function setMetaDescription($metaDescription)
    {
        $this->metaDescription = $metaDescription;

        return $this;
    }

    /**
     * Get metaDescription
     *
     * @return string
     */
    public function getMetaDescription()
    {
        return $this->metaDescription;
    }

    /**
     * Set metaKeywords
     *
     * @param string $metaKeywords
     * @return News
     */
    public function setMetaKeywords($metaKeywords)
    {
        $this->metaKeywords = $metaKeywords;

        return $this;
    }

    /**
     * Get metaKeywords
     *
     * @return string
     */
    public function getMetaKeywords()
    {
        return $this->metaKeywords;
    }

    /**
     * Set metaCanonical
     *
     * @param string $metaCanonical
     * @return News
     */
    public function setMetaCanonical($metaCanonical)
    {
        $this->metaCanonical = $metaCanonical;

        return $this;
    }

    /**
     * Get metaCanonical
     *
     * @return string
     */
    public function getMetaCanonical()
    {
        return $this->metaCanonical;
    }

    /**
     * Set metaImage
     *
     * @param string $metaImage
     * @return News
     */
    public function setMetaImage($metaImage)
    {
        $this->metaImage = $metaImage;

        return $this;
    }

    /**
     * Get metaImage
     *
     * @return string
     */
    public function getMetaImage()
    {
        return $this->metaImage;
    }

    /**
     * Set status
     *
     * @param string $status
     * @return News
     */
    public function setStatus($status)
    {
        $this->status = $status;

        return $this;
    }

    /**
     * Get status
     *
     * @return string
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Add category
     *
     * @param Category $category
     * @return News
     */
    public function addCategory(Category $category)
    {
        $this->categories[] = $category;

        return $this;
    }

    /**
     * Remove category
     *
     * @param Category $category
     */
    public function removeCategory(Category $category)
    {
        $this->categories->removeElement($category);
    }

    /**
     * Get categories
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getCategories()
    {
        return $this->categories;
    }

    /**
     * Set visitCounter
     *
     * @param integer $visitCounter
     * @return News
     */
    public function setVisitCounter($visitCounter)
    {
        $this->visitCounter = $visitCounter;

        return $this;
    }

    /**
     * Get visitCounter
     *
     * @return integer
     */
    public function getVisitCounter()
    {
        return $this->visitCounter;
    }

    /**
     * Set movie
     *
     * @param string $movie
     * @return News
     */
    public function setMovie($movie)
    {
        $this->movie = $movie;

        return $this;
    }

    /**
     * Get movie
     *
     * @return string
     */
    public function getMovie()
    {
        return $this->movie;
    }

    /**
     * Add image
     *
     * @param NewsImage $image
     * @return News
     */
    public function addImage(NewsImage $image)
    {
        $this->images[] = $image;

        return $this;
    }

    /**
     * Remove image
     *
     * @param NewsImage $image
     */
    public function removeImage(NewsImage $image)
    {
        $this->images->removeElement($image);
    }

    /**
     * Get images
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getImages()
    {
        return $this->images;
    }

    /**
     * Set type
     *
     * @param string $type
     * @return News
     */
    public function setType($type)
    {
        $this->type = $type;

        return $this;
    }

    /**
     * Get type
     *
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * Set enableDownload
     *
     * @param boolean $enableDownload
     * @return News
     */
    public function setEnableDownload($enableDownload)
    {
        $this->enableDownload = $enableDownload;

        return $this;
    }

    /**
     * Get enableDownload
     *
     * @return boolean
     */
    public function getEnableDownload()
    {
        return $this->enableDownload;
    }

    /**
     * Set author
     *
     * @param Administrator $author
     * @return News
     */
    public function setAutor(Administrator $author)
    {
        $this->author = $author;

        return $this;
    }

    /**
     * Get author
     *
     * @return Administrator
     */
    public function getAutor()
    {
        return $this->author;
    }

    /**
     * Set author
     *
     * @param Administrator $author
     * @return News
     */
    public function setAuthor(Administrator $author)
    {
        $this->author = $author;

        return $this;
    }

    /**
     * Get author
     *
     * @return Administrator
     */
    public function getAuthor()
    {
        return $this->author;
    }

    /**
     * Add tag
     *
     * @param Tag $tag
     * @return News
     */
    public function addTag(Tag $tag)
    {
        $this->tags[] = $tag;

        return $this;
    }

    /**
     * Remove tag
     *
     * @param Tag $tag
     */
    public function removeTag(Tag $tag)
    {
        $this->tags->removeElement($tag);
    }

    /**
     * Get tags
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getTags()
    {
        return $this->tags;
    }

    /**
     * Add user
     *
     * @param User $user
     * @return News
     */
    public function addUser(User $user)
    {
        $this->users[] = $user;

        return $this;
    }

    /**
     * Remove user
     *
     * @param User $user
     */
    public function removeUser(User $user)
    {
        $this->users->removeElement($user);
    }

    /**
     * Get users
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getUsers()
    {
        return $this->users;
    }

    /**
     * Add file
     *
     * @param File $file
     * @return News
     */
    public function addFile(File $file)
    {
        $this->files[] = $file;

        return $this;
    }

    /**
     * Remove file
     *
     * @param File $file
     */
    public function removeFile(File $file)
    {
        $this->files->removeElement($file);
    }

    /**
     * Get files
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getFiles()
    {
        return $this->files;
    }

    // ------------------------------------------------------------
    // Methods.
    // ------------------------------------------------------------

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->categories = new ArrayCollection();
        $this->files = new ArrayCollection();
        $this->images = new ArrayCollection();
        $this->tags = new ArrayCollection();
        $this->users = new ArrayCollection();
    }

    /**
     * Zwraca zdjęcie główne lub 'null', jeśli nie ma zdjęć..
     *
     * @return string|null
     */
    public function getImage()
    {
        if ($this->images->count() > 0) {
            return $this->images->first()->getName();
        }

        return null;
    }

    /**
     * Zwraca zdjęcie główne lub 'null', jeśli nie ma zdjęć..
     *
     * @deprecated getImageUrl() is deprecated since 19.10.2017. Use getImage() instead.
     * @return string|null
     */
    public function getImageUrl()
    {
        @trigger_error('getImageUrl() is deprecated since 19.10.2017. Use getImage() instead.', E_USER_DEPRECATED);

        return $this->getImage();
    }

    /**
     * Wstawienie krótkiego opisu na podstawie treści, jeśli krótki opis
     * pozostał pusty. Krótki opis jest generowany z oczyszczonej treści
     * z tagów HTML, skrócony do ostatniego całego słowa tak, aby nie
     * przekroczyć 500 znaków. Na końcu zawsze dodawane są 3 kropki.
     */
    public function setShortcutWhenIsNull()
    {
        if (empty($this->getShortcut())) {
            $this->setShortcut(
                preg_replace(
                    '/(.*)\s/',
                    '$1',
                    mb_substr(strip_tags($this->getContent()), 0, 500)
                ) . '...'
            );
        }
    }
}
