<?php

namespace Wi\Admin\NewsBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Wi\Admin\NewsBundle\Entity\Comment;
use Wi\Admin\NewsBundle\Entity\CommentGuest;
use Wi\Admin\NewsBundle\Entity\News;
use Wi\Admin\UserBundle\Entity\User;

/**
 * Comment
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2018 WEBimpuls Sp. z o.o.
 *
 * @ORM\Table(name="news_comments")
 * @ORM\Entity(repositoryClass="Wi\Admin\NewsBundle\Repository\CommentRepository")
 * @ORM\HasLifecycleCallbacks()
 * @Gedmo\SoftDeleteable(fieldName = "dateDeleted")
 */
class Comment
{
    /**
     * Status komentarza - aktywny.
     *
     * @const STATUS_ACTIVE
     */
    const STATUS_ACTIVE = 'A';

    /**
     * Status komentarza - usunięty.
     *
     * @const STATUS_DELETE
     */
    const STATUS_DELETE = 'D';

    /**
     * Status komentarza - ukryty.
     *
     * @const STATUS_HIDDEN
     */
    const STATUS_HIDDEN = 'H';

    /**
     * Status komentarza - spam.
     *
     * @const STATUS_SPAM
     */
    const STATUS_SPAM = 'S';

    /**
     * Status komentarza - czekuje na akceptację.
     *
     * @const STATUS_WAIT
     */
    const STATUS_WAIT = 'W';

    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var int
     *
     * @ORM\ManyToOne(targetEntity="Wi\Admin\NewsBundle\Entity\Comment", inversedBy="children")
     * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", nullable=true)
     */
    private $parent;

    /**
     * @var int
     *
     * @ORM\ManyToOne(targetEntity="Wi\Admin\NewsBundle\Entity\News")
     * @ORM\JoinColumn(name="news_id", referencedColumnName="id", nullable=false)
     */
    private $news;

    /**
     * @var int
     *
     * @ORM\ManyToOne(targetEntity="Wi\Admin\UserBundle\Entity\User")
     * @ORM\JoinColumn(name="user_id", referencedColumnName="id", nullable=true)
     */
    private $user;

    /**
     * @var int
     *
     * @ORM\ManyToOne(targetEntity="Wi\Admin\NewsBundle\Entity\CommentGuest", cascade={"persist"})
     * @ORM\JoinColumn(name="guest_id", referencedColumnName="id", nullable=true)
     */
    private $guest;

    /**
     * @var string
     *
     * @ORM\Column(name="content", type="text")
     */
    private $content;

    /**
     * @var string
     *
     * @ORM\Column(name="ip", type="string", length=32)
     */
    private $ip;

    /**
     * @var string
     *
     * @ORM\Column(name="host", type="string", length=255, nullable=true)
     */
    private $host;

    /**
     * @var string
     *
     * @ORM\Column(name="agent", type="string", length=128, nullable=true)
     */
    private $agent;

    /**
     * @var int
     *
     * @ORM\Column(name="likes", type="integer")
     */
    private $likes;

    /**
     * @var int
     *
     * @ORM\Column(name="dislikes", type="integer")
     */
    private $dislikes;

    /**
     * @var string
     *
     * @ORM\Column(name="status", type="string", length=1, options={"default": "A", "fixed" = true})
     */
    private $status;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateCreated", type="datetime")
     */
    private $dateCreated;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateDeleted", type="datetime", nullable=true)
     */
    private $dateDeleted;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\NewsBundle\Entity\Comment", mappedBy="parent", cascade={"persist"})
     * @ORM\OrderBy({"dateCreated" = "ASC"})
     */
    private $children;


    // ------------------------------------------------------------
    // Has Lifecycle Callbacks methods.
    // ------------------------------------------------------------

    /**
     * Funkcja wykonuje się podczas wstawiania nowego rekordu do bazy.
     *
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        // Ustawienie daty utworzenia.
        $this->dateCreated = new \DateTime();

        // Ustawienie liczby pozytywnych i negatywnych ocen.
        $this->likes = 0;
        $this->dislikes = 0;

        // Ustawienie statusu.
        if (is_null($this->status)) {
            $this->status = self::STATUS_ACTIVE;
        }
    }

    // ------------------------------------------------------------
    // Getters and setters methods.
    // ------------------------------------------------------------

    /**
     * Get id
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set parent
     *
     * @param Comment $parent
     * @return Comment
     */
    public function setParent(Comment $parent = null)
    {
        $this->parent = $parent;

        return $this;
    }

    /**
     * Get parent
     *
     * @return Comment
     */
    public function getParent()
    {
        return $this->parent;
    }

    /**
     * Set news
     *
     * @param News $news
     * @return Comment
     */
    public function setNews(News $news)
    {
        $this->news = $news;

        return $this;
    }

    /**
     * Get news
     *
     * @return News
     */
    public function getNews()
    {
        return $this->news;
    }

    /**
     * Set user
     *
     * @param User $user
     * @return Comment
     */
    public function setUser(User $user = null)
    {
        $this->user = $user;

        return $this;
    }

    /**
     * Get user
     *
     * @return User
     */
    public function getUser()
    {
        return $this->user;
    }

    /**
     * Set guest
     *
     * @param CommentGuest $guest
     * @return Comment
     */
    public function setGuest(CommentGuest $guest = null)
    {
        $this->guest = $guest;

        return $this;
    }

    /**
     * Get guest
     *
     * @return CommentGuest
     */
    public function getGuest()
    {
        return $this->guest;
    }

    /**
     * Set content
     *
     * @param string $content
     * @return Comment
     */
    public function setContent($content)
    {
        $this->content = $content;

        return $this;
    }

    /**
     * Get content
     *
     * @return string
     */
    public function getContent()
    {
        return $this->content;
    }

    /**
     * Set ip
     *
     * @param string $ip
     * @return Comment
     */
    public function setIp($ip)
    {
        $this->ip = $ip;

        return $this;
    }

    /**
     * Get ip
     *
     * @return string
     */
    public function getIp()
    {
        return $this->ip;
    }

    /**
     * Set host
     *
     * @param string $host
     * @return Comment
     */
    public function setHost($host)
    {
        $this->host = $host;

        return $this;
    }

    /**
     * Get host
     *
     * @return string
     */
    public function getHost()
    {
        return $this->host;
    }

    /**
     * Set agent
     *
     * @param string $agent
     * @return Comment
     */
    public function setAgent($agent)
    {
        $this->agent = $agent;

        return $this;
    }

    /**
     * Get agent
     *
     * @return string
     */
    public function getAgent()
    {
        return $this->agent;
    }

    /**
     * Set likes
     *
     * @param integer $likes
     * @return Comment
     */
    public function setLikes($likes)
    {
        $this->likes = $likes;

        return $this;
    }

    /**
     * Get likes
     *
     * @return int
     */
    public function getLikes()
    {
        return $this->likes;
    }

    /**
     * Set dislikes
     *
     * @param integer $dislikes
     * @return Comment
     */
    public function setDislikes($dislikes)
    {
        $this->dislikes = $dislikes;

        return $this;
    }

    /**
     * Get dislikes
     *
     * @return int
     */
    public function getDislikes()
    {
        return $this->dislikes;
    }

    /**
     * Set status
     *
     * @param string $status
     * @return Comment
     */
    public function setStatus($status)
    {
        $this->status = $status;

        return $this;
    }

    /**
     * Get status
     *
     * @return string
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Set dateCreated
     *
     * @param \DateTime $dateCreated
     * @return Comment
     */
    public function setDateCreated($dateCreated)
    {
        $this->dateCreated = $dateCreated;

        return $this;
    }

    /**
     * Get dateCreated
     *
     * @return \DateTime
     */
    public function getDateCreated()
    {
        return $this->dateCreated;
    }

    /**
     * Set dateDeleted
     *
     * @param \DateTime $dateDeleted
     * @return Comment
     */
    public function setDateDeleted($dateDeleted)
    {
        $this->dateDeleted = $dateDeleted;

        return $this;
    }

    /**
     * Get dateDeleted
     *
     * @return \DateTime
     */
    public function getDateDeleted()
    {
        return $this->dateDeleted;
    }

    /**
     * Add child
     *
     * @param Comment $child
     * @return Comment
     */
    public function addChild(Comment $child)
    {
        $this->children[] = $child;

        return $this;
    }

    /**
     * Remove child
     *
     * @param Comment $child
     */
    public function removeChild(Comment $child)
    {
        $this->children->removeElement($child);
    }

    /**
     * Get children
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getChildren()
    {
        return $this->children;
    }

    // ------------------------------------------------------------
    // Methods.
    // ------------------------------------------------------------

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->children = new ArrayCollection();
    }

    /**
     * Zwraca nazwę osoby, która napisała komentarz.
     *
     * @return string
     */
    public function getName() : string
    {
        if ($this->user) {
            return $this->user->getNickname();
        }

        if ($this->guest->getName()) {
            return $this->guest->getName();
        }

        return 'Anonim';
    }

    /**
     * Zwraca adres e-mail komentującego.
     *
     * @return string
     */
    public function getEmail() : string
    {
        if ($this->user) {
            return $this->user->getEmail();
        }

        return $this->guest->getEmail();
    }

    /**
     * Zwraca link do avataru osoby komentującej.
     *
     * @return string
     */
    public function getAvatar() : string
    {
        if ($this->user) {
            if ($avatar = $this->user->getAvatarLink()) {
                return $avatar;
            }
        }

        return 'media/Public/img/guest.jpg';
    }
}
