<?php

namespace Wi\Admin\NewsBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\NewsBundle\Entity\Category;
use Wi\Admin\NewsBundle\Entity\News;
use Wi\Admin\NewsBundle\Entity\NewsImage;
use Wi\Admin\NewsBundle\Entity\Tag;
use Wi\Admin\NewsBundle\Form\NewsDeleteType;
use Wi\Admin\NewsBundle\Form\NewsImageType;
use Wi\Admin\NewsBundle\Form\NewsType;
use Wi\Admin\NewsBundle\Service\Pdf;

/**
 * News controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class NewsController extends Controller
{
    /**
     * Lists all news entities.
     *
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction(Request $request)
    {
        $route = $request->attributes->get('_route');
        $type = News::TYPE_NEWS;
        $typeCategory = Category::TYPE_NEWS;
        if ($route == 'wi_admin_blog_index') {
            $type = News::TYPE_BLOG;
            $typeCategory = Category::TYPE_BLOG;
        }
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie wszystkich aktywnych aktualności.
        $paginator = $this->get('knp_paginator');
        $news = $paginator->paginate(
            $em->getRepository('WiAdminNewsBundle:News')
                ->getFindAllActiveQuery(
                    $type,
                    $request->query->getInt('category_id'),
                    $request->query->get('q')
                ),
            $request->query->getInt('page', 1),
            $request->query->getInt('per_page') ?: 10
        );

        // Pobranie wszystkich kategorii.
        $categories = $em->getRepository('WiAdminNewsBundle:Category')
            ->findAllActiveWithoutParent($typeCategory)
        ;

        if ($route == 'wi_admin_blog_index') {
            return $this->render('WiAdminNewsBundle:Blog:index.html.php', [
                'categories' => $categories,
                'news' => $news,
            ]);
        }
        return $this->render('WiAdminNewsBundle:News:index.html.php', [
            'categories' => $categories,
            'news' => $news,
        ]);
    }

    /**
     * Create a new news entity.
     *
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request)
    {
        $route = $request->attributes->get('_route');
        $type = News::TYPE_NEWS;
        $typeCategory = Category::TYPE_NEWS;
        if ($route == 'wi_admin_blog_new') {
            $type = News::TYPE_BLOG;
            $typeCategory = Category::TYPE_BLOG;
        }

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Utworzenie nowego obiektu aktualności.
        $news = new News();

        // Ustawienie typu newsa jako News.
        $news->setType($type);
        $news->setAutor($this->getUser());

        // Dodanie domyślnej kategorii, jeśli została przekazana oraz istnieje.
        if ($request->query->getInt('category_id')) {
            $category = $em->getRepository('WiAdminNewsBundle:Category')
                ->findOneById($request->query->getInt('category_id'))
            ;

            if (! is_null($category)) {
                $news->addCategory($category);
            }
        }

        // Ustawienie domyślnej daty publikacji.
        $news->setDateOfPublication(new \DateTime());

        // Generowanie formularza.
        $form = $this->createForm(NewsType::class, $news, ['type' => $typeCategory]);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {

            // Zapis.
            $em->persist($news);
            $em->flush();

            if ($route == 'wi_admin_blog_new') {
                return $this->redirectToRoute('wi_admin_blog_show', [
                    'id' => $news->getId(),
                ]);
            }
            return $this->redirectToRoute('wi_admin_news_show', [
                'id' => $news->getId(),
            ]);
        }

        if ($route == 'wi_admin_blog_new') {
            return $this->render('WiAdminNewsBundle:Blog:new.html.php', [
                'form' => $form->createView(),
                'news' => $news,
            ]);
        }
        return $this->render('WiAdminNewsBundle:News:new.html.php', [
            'form' => $form->createView(),
            'news' => $news,
        ]);
    }

    /**
     * Edit a news entity.
     *
     * @param   Request $request
     * @param   News $news
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, News $news)
    {
        $route = $request->attributes->get('_route');
        $type = News::TYPE_NEWS;
        $typeCategory = Category::TYPE_NEWS;
        if ($route == 'wi_admin_blog_edit') {
            $type = News::TYPE_BLOG;
            $typeCategory = Category::TYPE_BLOG;
        }

        // Generowanie formularza.
        $form = $this->createForm(NewsType::class, $news, ['type' => $typeCategory]);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Zapis.
            $em->persist($news);
            $em->flush();

            if ($route == 'wi_admin_blog_edit') {
                return $this->redirectToRoute('wi_admin_blog_show', [
                    'id' => $news->getId(),
                ]);
            }
            return $this->redirectToRoute('wi_admin_news_show', [
                'id' => $news->getId(),
            ]);
        }

        $newsImage = new NewsImage();
        $newsImage->setNews($news);
        $imageForm = $this->createForm(NewsImageType::class, $newsImage);
        $imageForm->handleRequest($request);

        if ($imageForm->isSubmitted() && $imageForm->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Upload file.
            if ($imageForm->get('file')->getData()) {

                $newsUpload = $this->get('wi.admin.news.upload');
                if ($route == 'wi_admin_blog_edit') {
                    $newsUpload = $this->get('wi.admin.blog.upload');
                    $newsUpload = $this->get('wi.admin.blog.upload');
                }

                $filename = $newsUpload->upload(
                    $imageForm->get('file')->getData(),
                    [
                        'crop_x' => (int) $imageForm->get('crop_x')->getData(),
                        'crop_y' => (int) $imageForm->get('crop_y')->getData(),
                        'crop_height' => (int) $imageForm->get('crop_height')->getData(),
                        'crop_width' => (int) $imageForm->get('crop_width')->getData(),
                        'cropSave' => (bool) $imageForm->get('cropSave')->getData(),
                    ]
                );

                // Set filename.
                $newsImage->setName($filename);
            }

            // Zapis.
            $em->persist($newsImage);
            $em->flush();

            if ($route == 'wi_admin_blog_edit') {
                return $this->redirectToRoute('wi_admin_blog_edit', [
                    'id' => $news->getId(),
                ]);
            }

            return $this->redirectToRoute('wi_admin_news_edit', ['id' => $news->getId()]);
        }

        if ($route == 'wi_admin_blog_edit') {
            return $this->render('WiAdminNewsBundle:Blog:edit.html.php', [
                'form' => $form->createView(),
                'imageForm' => $imageForm->createView(),
                'news' => $news,
            ]);
        }
        return $this->render('WiAdminNewsBundle:News:edit.html.php', [
            'form' => $form->createView(),
            'imageForm' => $imageForm->createView(),
            'news' => $news,
        ]);
    }

    /**
     * Delete a news entity.
     *
     * @param   Request $request
     * @param   News $news
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, News $news)
    {
        $route = $request->attributes->get('_route');

        // Generowanie formularza.
        $deleteForm = $this->createForm(NewsDeleteType::class);
        $deleteForm->handleRequest($request);

        // Weryfikacja czy formularz został przesłany i pomyślnie zwalidowany.
        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Usunięcie.
            $em->remove($news);
            $em->flush();

            if ($route == 'wi_admin_blog_delete') {
                return $this->redirectToRoute('wi_admin_blog_index');
            }
            return $this->redirectToRoute('wi_admin_news_index');
        }

        if ($route == 'wi_admin_blog_delete') {
            return $this->render('WiAdminNewsBundle:Blog:delete.html.php', [
                'deleteForm' => $deleteForm->createView(),
                'news' => $news,
            ]);
        }
        return $this->render('WiAdminNewsBundle:News:delete.html.php', [
            'deleteForm' => $deleteForm->createView(),
            'news' => $news,
        ]);
    }

    /**
     * Show a news entity.
     *
     * @param   Request $request
     * @param   News $news
     */
    public function showAction(News $news, Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $campaigns = $em->getRepository('WiAdminNewsLetterBundle:Campaign')->findAllActive();

        $route = $request->attributes->get('_route');
        $template = 'WiAdminNewsBundle:News:show.html.php';

        if ($route == 'wi_admin_blog_show') {
            $template = 'WiAdminNewsBundle:Blog:show.html.php';
        }
        return $this->render($template, [
            'campaigns' => $campaigns,
            'files' => $news->getFiles(),
            'news' => $news,
        ]);
    }

    /**
     * Aktualizacja kolejności zdjęć.
     *
     * @param Request $request
     * @param News $news
     * @return JsonResponse
     */
    public function updateImageOrderAction(Request $request, News $news)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        $ids = $request->request->get('ids');
        $images = $news->getImages();

        foreach ($images as $image) {
            $image->setOrderNr(array_search($image->getId(), $ids));
            $em->persist($image);
        }

        $em->flush();

        return new JsonResponse('OK');
    }

    /**
     * Usuwanie zdjeć.
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function deleteImageAction(NewsImage $newsImage, Request $request)
    {
        $route = $request->attributes->get('_route');

        $news_id = $newsImage->getNews()->getId();

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        $em->remove($newsImage);
        $em->flush();

        if ($route == 'wi_admin_blog_delete_image') {
            return $this->redirectToRoute('wi_admin_blog_edit', ['id' => $news_id]);
        }
        return $this->redirectToRoute('wi_admin_news_edit', ['id' => $news_id]);
    }

    /**
     * Dodanie nowego tagu do bazy.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function addNewTagAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Nazwa nowego tagu.
        $name = $ids = $request->get('tagName');

        // Sprawdza czy taki tag istniej w bazie.
        $tagExist = $em->getRepository('WiAdminNewsBundle:Tag')->findOneByName($name);

        if (is_null($tagExist)) {
            $tag = new Tag();
            $tag->setName($name);

            // Zapis.
            $em->persist($tag);
            $em->flush();

            $id = $tag->getId();
        } else {
            $id = $tagExist->getId();
        }
        return new JsonResponse($id);
    }

    /**
     * Obsługa edycji na żywo treści aktualności.
     *
     * @param Request $request
     * @param News $news
     * @return JsonResponse
     */
    public function liveEditAction(Request $request, News $news)
    {
        $newsJson = $request->request->get('news');

        if (empty($newsJson)) {
            return new JsonResponse([
                'message' => 'Empty params',
                'status' => false,
            ]);
        }

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Aktualizacja zawartości strony.
        if (! empty($content = $newsJson['content'])) {
            $news->setContent(json_decode($content));
        }

        $em->persist($news);
        $em->flush($news);

        return new JsonResponse([
            'status' => true,
        ]);
    }

    /**
     * Zapisanie aktualności/wpisu do PDF i pobranie.
     *
     * @param Pdf $pdf
     * @param Request $request
     * @param News $news
     * @return BinaryFileResponse
     */
    public function pdfAction(Pdf $pdf, Request $request, News $news)
    {
        return $this->file($pdf->generate($pdf->setNews($news)->download()->html()));
    }
}
