<?php

namespace Wi\Admin\MenuBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Wi\Admin\MenuBundle\Entity\ItemMenu;
use Wi\Admin\MenuBundle\Entity\Location;
use Wi\Admin\MenuBundle\Form\ItemMenuDeleteType;
use Wi\Admin\MenuBundle\Form\ItemMenuType;
use Wi\Admin\PageBundle\Entity\Page;

/**
 * Menu controller.
 *
 * @author Piotr Krzok <piotr.krzok@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class MenuController extends Controller
{
    /**
     * List of all navigations.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie lokalizacji menu.
        $locations = $em->getRepository('WiAdminMenuBundle:Location')
            ->findAllActive()
        ;

        return $this->render('WiAdminMenuBundle:Menu:index.html.php', [
            'locations' => $locations,
        ]);
    }

    /**
     * Creates a new menu entity.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Utworzenie nowego obiektu menu.
        $item = new ItemMenu();

        // Dodanie domyślnej lokalizacji, jeśli została przekazana oraz istnieje.
        if ($request->query->getInt('location_id')) {
            $location = $em->getRepository('WiAdminMenuBundle:Location')
                ->findOneById($request->query->getInt('location_id'))
            ;

            if (! is_null($location)) {
                $item->setLocation($location);
            }
        }

        // Dodanie rodzica, jeśli został przekazany oraz istnieje.
        if ($request->query->getInt('parent_id')) {
            $item->setParent($em->getRepository('WiAdminMenuBundle:ItemMenu')
                ->findOneById($request->query->getInt('parent_id'))
            );
        }

        // Generowanie formularza.
        $form = $this->createForm(ItemMenuType::class, $item);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Zapis.
            $em->persist($item);
            $em->flush();

            return $this->redirectToRoute('wi_admin_menu_index');
        }

        return $this->render('WiAdminMenuBundle:Menu:new.html.php', [
            'form' => $form->createView(),
            'item' => $item,
        ]);
    }

    /**
     * Edit a itemMenu entity.
     *
     * @param Request $request
     * @param ItemMenu $itemMenu
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, ItemMenu $itemMenu)
    {
        // Generowanie formularza.
        $form = $this->createForm(ItemMenuType::class, $itemMenu);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Zapis.
            $em->persist($itemMenu);
            $em->flush();

            return $this->redirectToRoute('wi_admin_menu_index');
        }

        return $this->render('WiAdminMenuBundle:Menu:edit.html.php', [
            'form' => $form->createView(),
            'itemMenu' => $itemMenu,
        ]);
    }

    /**
     * Delete a itemMenu entity.
     *
     * @param Request $request
     * @param ItemMenu $itemMenu
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, ItemMenu $itemMenu)
    {
        // Utworzenie formularza usuwania grup stron.
        $formDelete = $this->createForm(ItemMenuDeleteType::class, $itemMenu);
        $formDelete->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($formDelete->isSubmitted() && $formDelete->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            if ($itemMenu->getChildren()->count()) {
                // Usuwanie wszystkich dzieci.
                if ($formDelete->get('deleteAll')->getData()) {
                    // Rekursywne usuwanie grup.
                    $recursiveRemove = function ($itemMenu) use (&$recursiveRemove, $em) {
                        if ($itemMenu->getChildren()->count() > 0) {
                            foreach ($itemMenu->getChildren() as $child) {
                                $recursiveRemove($child);
                            }
                        }

                        $em->remove($itemMenu);
                    };

                    $recursiveRemove($itemMenu);
                } else {
                    // Usunięcie pozycji i przenoszenie pozycji.
                    $newItemMenu = $formDelete->get('newItemMenu')->getData();

                    // Przypisanie nowego rodzica pozycjom menu.
                    foreach ($itemMenu->getChildren() as $child) {
                        // Ustawienie nowego rodzica.
                        $child->setParent($newItemMenu);

                        // Przygotowanie do zapisu.
                        $em->persist($child);
                    }

                    // Zapis edytowanych dzieci.
                    $em->flush();
                }
            }

            // Usuwanie pozycji.
            $em->remove($itemMenu);
            $em->flush();

            // Ustawienie daty usunięcia w grup ze statusem 'D'. Przy usuwaniu
            // jest problem ustawić dlatego ten kod się pojawił.
            $deletedItemMenus = $em->getRepository('WiAdminMenuBundle:ItemMenu')
                ->findBy([
                    'dateDeleted' => null,
                    'status' => 'D',
                ])
            ;

            if (count($deletedItemMenus) > 0) {
                foreach ($deletedItemMenus as $deletedItemMenu) {
                    $deletedItemMenu->setDateDeleted(new \DateTime());
                    $em->persist($deletedItemMenu);
                }

                $em->flush();
            }

            return $this->redirectToRoute('wi_admin_menu_index');
        }

        return $this->render('WiAdminMenuBundle:Menu:delete.html.php', array(
            'formDelete' => $formDelete->createView(),
            'itemMenu' => $itemMenu,
        ));
    }

    /**
     * Widok HTML dla Ajaxa.
     *
     * @param Request $request
     * @return Response
     */
    public function generateMenuViewAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie tabeli itemModule
        $itemModule = $em->getRepository('WiAdminMenuBundle:ItemModule')
            ->findAll()
        ;

        $translator = $this->get('translator');
        $permission = $this->get('security.authorization_checker');

        ob_start();
        include $this->getParameter('kernel.root_dir').'/../src/Wi/Admin/MenuBundle/Resources/views/Menu/generateMenu.html.php';
        $html = ob_get_clean();

        return new Response($html);
    }

    /**
     * Ajax list pages.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function pagesListAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Utworzenie pustej tabeli wyników
        $out = [];

        // Sprawdzenie jaki moduł został wybrany.
        if ($itemModule = $request->request->getInt('itemModule')) {

            switch ($itemModule) {
                case 1: // Zwrócenie wyniku dla strony głównej.
                    $out = [];
                    break;
                case 2: // Zwrócenie wyniku dla Stron.
                    $pages = $em->getRepository('WiAdminPageBundle:Page')
                        ->findAll()
                    ;

                    foreach ($pages as $page) {
                        $out[] = [
                            'id' => $page->getId(),
                            'name' => $page->getTitle(),
                        ];
                    }
                    break;
                case 3: // Zwrócenie wyniku dla Aktualności.
                    $out = [];
                    break;
                case 4: // Zwrócenie wyniku dla Kategorii.
                    $categories = $em->getRepository('WiAdminNewsBundle:Category')
                        ->findAll()
                    ;

                    foreach ($categories as $category) {
                        $out[] = [
                            'id' => $category->getId(),
                            'name' => $category->getName(),
                        ];
                    }
                    break;
                case 5: // Zwrócenie wyniku dla Newsów.
                    $news = $em->getRepository('WiAdminNewsBundle:News')
                        ->findAll()
                    ;

                    foreach ($news as $newsItem) {
                        $out[] = [
                            'id' => $newsItem->getId(),
                            'name' => $newsItem->getTitle(),
                        ];
                    }
                    break;
            }
        }

        return new JsonResponse($out);
    }

    /**
     * Ajax generate link.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function ajaxGenerateUrlAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Utworzenie instantcji Slugger.
        $slugger = $this->get('front.slugger');

        // Utworzenie pustej tabeli wyników
        $out = [];

        // Sprawdzenie jaki moduł został wybrany.
        if ($itemModule = $request->request->getInt('itemModule')) {
            $route = $em
                ->getRepository('WiAdminMenuBundle:ItemModule')
                ->findOneById($itemModule)
            ;

            switch ($itemModule) {
                case 1: // Zwrócenie linku dla strony głównej.
                    $out = [
                        'url' => $this->generateUrl($route->getRoute()),
                    ];
                    break;
                case 2: // Zwrócenie linku dla strony.
                    $page = $em
                        ->getRepository('WiAdminPageBundle:Page')
                        ->findOneById($request->request->getInt('itemPages'))
                    ;
                    $out = [
                        'url' => $this->generateUrl($route->getRoute(), [
                            'id' => $page->getId(),
                            'slug' => $slugger->slugify($page->getTitle()),
                        ]),
                    ];
                    break;
                case 3: // Zwrócenie linku dla Aktualności.
                    $out = [
                        'url' => $this->generateUrl($route->getRoute()),
                    ];
                    break;
                case 4: // Zwrócenie linku dla Kategorii.
                    $page = $em
                        ->getRepository('WiAdminNewsBundle:Category')
                        ->findOneById($request->request->getInt('itemPages'))
                    ;
                    $out = [
                        'url' => $this->generateUrl($route->getRoute(), [
                            'id' => $page->getId(),
                            'slug' => $slugger->slugify($page->getName()),
                        ]),
                    ];
                    break;
                case 5: // Zwrócenie linku dla Newsów.
                    $page = $em
                        ->getRepository('WiAdminNewsBundle:News')
                        ->findOneById( $request->request->getInt('itemPages'))
                    ;
                    $out = [
                        'url' => $this->generateUrl($route->getRoute(), [
                            'id' => $page->getId(),
                            'slug' => $slugger->slugify($page->getTitle()),
                        ]),
                    ];
                    break;
            }
        }

        return new JsonResponse($out);
    }

    /**
     * Ajax save list parameters.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function ajaxSaveListParametersAction(Request $request)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Utworzenie pustej tabeli wyników
        $out = [];

        // Pobranie danych json ze skryptu js nestable.
        if ($daneJson = $request->request->get('daneJson')) {
            // Dekodowanie danych json do tablicy.
            $json_decode = json_decode($daneJson, true);

            // Przetwarzanie tablicy.
            foreach ($json_decode as $key => $item) {
                // Pobranie aktualnie przetwarzanego elementu menu z bazy
                $itemMenu = $em
                    ->getRepository('WiAdminMenuBundle:ItemMenu')
                    ->findOneById($item['id'])
                ;
                $itemMenu->setOrderNr($key);
                $itemMenu->setParent(null);

                // Przygotowanie do zapisu.
                $em->persist($itemMenu);

                // Jeżeli są dzieci.
                if (isset($item['children'])) {
                    $this->childrenMenuList($item);
                }
            }

            // Zapis edytowanych pozycji.
            $em->flush();
        }

        $out[] = 'Zapisano';

        return new JsonResponse($out);
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     *
     *
     * @param array $arr
     */
    private function childrenMenuList(array $arr)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie danych rodzica.
        $itemRodzic = $em
            ->getRepository('WiAdminMenuBundle:ItemMenu')
            ->findOneById($arr['id'])
        ;

        foreach ($arr['children'] as $key => $child){
            // Pobranie aktualnie przetwarzanego elementu menu z bazy.
            $itemMenu = $em
                ->getRepository('WiAdminMenuBundle:ItemMenu')
                ->findOneById($child['id'])
            ;

            // Ustawienie parametrów.
            $itemMenu
                ->setOrderNr($key)
                ->setParent($itemRodzic)
            ;

            // Przygotowanie do zapisu.
            $em->persist($itemMenu);

            // Jeżeli są kolejne dzieci.
            if (isset($child['children'])) {
                $this->childrenMenuList($child);
            }
        }
    }
}
