<?php

namespace Wi\Admin\MediaBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\Filesystem\Exception\IOExceptionInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\HttpFoundation\File\File;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Wi\Admin\MediaBundle\Form\AddFolderType;
use Wi\Admin\MediaBundle\Form\DeleteType;
use Wi\Admin\MediaBundle\Form\RenameType;

/**
 * Media controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class MediaController extends Controller
{
    /**
     * Media manager default action.
     *
     * @param   Request $request
     * @param   string  $directory
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction(Request $request, $directory = null)
    {
        // Set media url.
        $mediaUrl = $request->getScheme() . '://' .
            $request->getHttpHost() . $request->getBasePath() .
            '/media/assets/'
        ;

        // Get media uploads utils.
        $mediaUploadUtils = $this->get('wi.admin.media.upload');

        $directory = preg_replace('/(\/+|\\+)/', '/', $directory);
        $currentDirectory = $this->getParameter('assets_dir') . $directory;
        $previousDirectory = rtrim(
            str_replace(
                $this->getParameter('assets_dir'),
                '',
                str_replace(
                    preg_replace("/(.*)\//", '', rtrim($directory, '/')),
                    '',
                    $currentDirectory
                )
            ),
            '/'
        );

        // Directories.
        $directories = [];
        // Files.
        $files = [];

        $fs = new Filesystem();

        if ($fs->exists($currentDirectory)) {
            $finder = new Finder();
            $finder
                ->directories()
                ->in($currentDirectory)
                ->depth(0)
            ;

            // Collect directories.
            foreach ($finder as $dir) {
                $directories[] = $dir;
            }

            $finder = new Finder();
            $finder
                ->files()
                ->depth(0)
                ->in($currentDirectory)
            ;

            // Collect files.
            foreach ($finder as $file) {
                $File = new File($file->getRealPath());
                $File->dimensions = $mediaUploadUtils->getDimensions(
                    $File->getRealPath(),
                    $File->getExtension()
                );
                $File->directory = $directory;
                $File->link = $mediaUrl . $directory . $File->getFilename();
                $File->size = $mediaUploadUtils->formatBytes($File->getSize());
                $files[] = $File;
            }
        }

        // Generowanie formularza.
        $addFolderForm = $this->createForm(AddFolderType::class);
        $addFolderForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($addFolderForm->isSubmitted() && $addFolderForm->isValid()) {
            $dirname = $mediaUploadUtils->prepareDirname(
                $addFolderForm->get('name')->getData()
            );
            $fs->mkdir($currentDirectory . $dirname);

            return $this->redirect($request->server->get('REQUEST_URI'));
        }

        // Generowanie formularza.
        $deleteForm = $this->createForm(DeleteType::class);
        $deleteForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($deleteForm->isSubmitted() && $deleteForm->isValid()) {
            $delete = $deleteForm->get('delete')->getData();
            $fs->remove($currentDirectory . $delete);

            return $this->redirect($request->server->get('REQUEST_URI'));
        }

        // Generowanie formularza.
        $renameForm = $this->createForm(RenameType::class);
        $renameForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($renameForm->isSubmitted() && $renameForm->isValid()) {
            $oldName = $renameForm->get('oldName')->getData();
            $name = $renameForm->get('name')->getData();

            if (strpos($oldName, '.') === false) {
                $name = $mediaUploadUtils->getEnsureDirnameUnique(
                    $currentDirectory,
                    $mediaUploadUtils->prepareDirname($name)
                );
            } else {
                $name = $mediaUploadUtils->getEnsureFilenameUnique(
                    $currentDirectory,
                    $mediaUploadUtils->prepareFilename(
                        $name,
                        preg_replace('/^.*\./', '', $oldName)
                    )
                );
            }

            $fs->rename(
                $currentDirectory . $oldName,
                $currentDirectory . $name,
                true
            );

            return $this->redirect($request->server->get('REQUEST_URI'));
        }

        return $this->render('WiAdminMediaBundle:Media:index.html.php', [
            'addFolderForm' => $addFolderForm->createView(),
            'currentDirectory' => $currentDirectory,
            'deleteForm' => $deleteForm->createView(),
            'directories' => $directories,
            'files' => $files,
            'previousDirectory' => $previousDirectory,
            'renameForm' => $renameForm->createView(),
        ]);
    }

    /**
     * Upload Action.
     *
     * @param   Request $request
     * @return  JsonResponse
     */
    public function uploadAction(Request $request)
    {
        // Get media uploads utils.
        $mediaUploadUtils = $this->get('wi.admin.media.upload');

        $d = $request->query->get('d');
        $fs = new Filesystem();

        // Sprawdzenie, czy istnieją pliki.
        if (empty($request->files) || $request->files->get('file')->getError()) {
            if ($request->query->get('ref')) {
                return $this->redirect($request->headers->get('referer'));
            }

            return new JsonResponse([
                'OK' => 0,
                'info' => 'Failed to move uploaded file.',
            ]);
        }

        // Sprawdzenie, czy katalog istnieje.
        if (! is_dir($this->getParameter('assets_dir').$d)) {
            // Tworzy katalog tmp/.
            $fs->mkdir($this->getParameter('assets_dir').$d);
        }

        // Ustawienie liczby chunków.
        $chunk = ! empty($request->request->get('chunk')) ? (int) $request->request->get('chunk') : 0;
        $chunks = ! empty($request->request->get('chunks')) ? (int) $request->request->get('chunks') : 0;

        // Ustawienie nazwy pliku.
        $fileName = ! empty($request->request->get('name')) ?
            strtolower($request->request->get('name')) :
            strtolower($request->files->get('file')->getClientOriginalName())
        ;

        if (! $mediaUploadUtils->isAllowedExt($fileName)) {
            if ($request->query->get('ref')) {
                return $this->redirect($request->headers->get('referer'));
            }

            return new JsonResponse([
                'OK' => 0,
                'info' => 'File extension is not allowed.',
            ]);
        }

        // Ustawienie ścieżki do pliku.
        $filePath = $this->getParameter('assets_dir') . $d . $fileName;

        // Otworzenie pliku tymczasowego.
        $out = @fopen("{$filePath}.part", $chunk == 0 ? "wb" : "ab");

        if ($out) {
            // Read binary input stream and append it to temp file
            $in = @fopen($request->files->get('file')->getPathName(), "rb");

            if ($in) {
                while ($buff = fread($in, 4096)) {
                    fwrite($out, $buff);
                }
            } else {
                if ($request->query->get('ref')) {
                    return $this->redirect($request->headers->get('referer'));
                }

                return new JsonResponse([
                    'OK' => 0,
                    'info' => 'Failed to open input stream.',
                ]);
            }

            @fclose($in);
            @fclose($out);

            @unlink($request->files->get('file')->getPathName());
        } else {
            if ($request->query->get('ref')) {
                return $this->redirect($request->headers->get('referer'));
            }

            return new JsonResponse([
                'OK' => 0,
                'info' => 'Failed to open output stream.',
            ]);
        }

        // Check if file has been uploaded
        if (! $chunks || $chunk == $chunks - 1) {
            // Prepare filename.
            $filename = $mediaUploadUtils->getEnsureFilenameUnique(
                str_replace(
                    basename($filePath),
                    '',
                    $filePath
                ),
                $mediaUploadUtils->prepareFilename(
                    basename($filePath),
                    preg_replace('/^.*\./', '', basename($filePath))
                )
            );

            // New file path.
            $newFilePath = str_replace(basename($filePath), '', $filePath) . $filename;

            // Strip the temp .part suffix off
            rename("{$filePath}.part", $newFilePath);
        }

        if ($request->query->get('ref')) {
            return $this->redirect($request->headers->get('referer'));
        }

        return new JsonResponse([
            'OK' => 1,
            'info' => 'Upload successful.',
        ]);
    }

    /**
     * Ajax list image.
     *
     * @return JsonResponse
     */
    public function imageListAction()
    {
        // Get Filesystem component.
        $fs = new Filesystem();

        // Create assets dir if not exists.
        $fs->mkdir($this->getParameter('assets_dir'));

        // Output array.
        $out = [];

        // Get assets helper.
        $assetsHelper = $this->get('templating.helper.assets');

        // Get finder component.
        $finder = new Finder();
        $finder
            ->files()
            ->in($this->getParameter('assets_dir'))
            ->name('/\.jpg|\.jpeg|\.png|\.gif/')
        ;

        foreach ($finder as $file) {
            $title = ltrim(
                preg_replace(
                    '/\\\/',
                    '/',
                    str_replace(
                        $this->getParameter('assets_dir'),
                        '',
                        $file->getPathname()
                    )
                ),
                '/'
            );
            $value = $assetsHelper->getUrl('media/assets/' . $title);
            $out[] = [
                'title' => $title,
                'value' => $value,
            ];
        }

        return new JsonResponse($out);
    }

    /**
     * Widok HTML dla Ajaxa.
     *
     * @param   Request $request
     * @return  Response
     */
    public function tinymceMediaViewAction(Request $request)
    {
        // Get media uploads utils.
        $mediaUploadUtils = $this->get('wi.admin.media.upload');
        $fs = new Filesystem();

        // Set media url.
        $url = $request->getScheme() . '://' .
            $request->getHttpHost() . $request->getBasePath()
        ;
        $mediaUrl = $url . '/media/assets/';
        $scriptUrl = $url . '/media/';

        $directory = $request->query->get('directory');

        $directory = rtrim(
            ltrim(
                preg_replace(
                    '/\\\|\//',
                    '/',
                    preg_replace(
                        '/\./',
                        '',
                        $directory
                    )
                ),
                '/'
            ),
            '/'
        ) . '/';

        $currentDirectory = $this->getParameter('assets_dir') . $directory;
        $previousDirectory = rtrim(
            str_replace(
                preg_replace(
                    "/(.*)\//",
                    '',
                    rtrim($directory, '/')
                ),
                '',
                $directory
            ),
            '/'
        );

        // Obsługa formularzy.
        if ($request->request->get('rename')) {
            $oldName = trim($request->get('oldName'));
            $name = trim($request->get('name'));

            if (strpos($oldName, '.') === false) {
                $name = $mediaUploadUtils->getEnsureDirnameUnique(
                    $currentDirectory,
                    $mediaUploadUtils->prepareDirname($name)
                );
            } else {
                $name = $mediaUploadUtils->getEnsureFilenameUnique(
                    $currentDirectory,
                    $mediaUploadUtils->prepareFilename(
                        $name,
                        preg_replace('/^.*\./', '', $oldName)
                    )
                );
            }

            $fs->rename(
                $currentDirectory . $oldName,
                $currentDirectory . $name,
                true
            );
        }

        if ($request->request->get('delete')) {
            $delete = $request->request->get('delete');
            $fs->remove($currentDirectory . $delete);
        }

        if ($request->request->get('add_folder')) {
            $dirname = $mediaUploadUtils->prepareDirname(
                trim($request->request->get('name'))
            );
            $fs->mkdir($currentDirectory . $dirname);
        }

        // Directories.
        $directories = [];

        // Files.
        $files = [];

        if ($fs->exists($currentDirectory)) {
            $finder = new Finder();
            $finder
                ->directories()
                ->in($currentDirectory)
                ->depth(0)
            ;

            // Collect directories.
            foreach ($finder as $dir) {
                $dir->directory = str_replace(
                    $this->getParameter('assets_dir'),
                    '',
                    $currentDirectory
                ) . $dir->getFilename();
                $dir->link = $mediaUrl . $directory . $dir->getFilename();
                $directories[] = $dir;
            }

            $finder = new Finder();
            $finder
                ->files()
                ->depth(0)
                ->in($currentDirectory)
                ->name('/\.jpg|\.jpeg|\.png|\.gif/')
            ;

            // Collect files.
            foreach ($finder as $file) {
                $File = new File($file->getRealPath());
                $File->dimensions = $mediaUploadUtils->getDimensions(
                    $File->getRealPath(),
                    $File->getExtension()
                );
                $File->directory = $directory;
                $File->link = $mediaUrl . $directory . $File->getFilename();
                $File->size = $mediaUploadUtils->formatBytes($File->getSize());
                $files[] = $File;
            }
        }

        $translator = $this->get('translator');
        $permission = $this->get('security.authorization_checker');

        ob_start();
        include $this->getParameter('kernel.root_dir').'/../src/Wi/Admin/MediaBundle/Resources/views/Media/tinymce.html.php';
        $html = ob_get_clean();

        return new Response($html);
    }
}
