<?php

namespace Wi\Admin\FAQBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\FAQBundle\Entity\Category;
use Wi\Admin\FAQBundle\Form\CategoryDeleteType;
use Wi\Admin\FAQBundle\Form\CategoryType;

/**
 * Category controller.
 *
 * @author Maksymilian Baryczka <mmaxioo33@gmail.com>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class CategoryController extends Controller
{
    /**
     * Lists all category entities.
     *
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction()
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();

        // Pobranie wszystkich aktywnych kategorii.
        $categories = $em->getRepository('WiAdminFAQBundle:Category')
            ->findAllActive()
        ;

        return $this->render('WiAdminFAQBundle:Category:index.html.php', [
            'categories' => $categories,
        ]);
    }

    /**
     * Creates a new category entity.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request)
    {
        // Utworzenie nowego obiektu kategorii.
        $category = new Category();

        // Generowanie formularza.
        $form = $this->createForm(CategoryType::class, $category);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Zapis.
            $em->persist($category);
            $em->flush();

            return $this->redirectToRoute('wi_admin_faq_category_show', [
                'id' => $category->getId(),
            ]);
        }

        return $this->render('WiAdminFAQBundle:Category:new.html.php', [
            'category' => $category,
            'form' => $form->createView(),
        ]);
    }

    /**
     * Finds and displays a category entity.
     *
     * @param Category $category
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(Category $category)
    {
        return $this->render('WiAdminFAQBundle:Category:show.html.php', [
            'category' => $category,
        ]);
    }

    /**
     * Displays a form to edit an existing category entity.
     *
     * @param Request $request
     * @param Category $category
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Category $category)
    {
        // Generowanie formularza.
        $editForm = $this->createForm(CategoryType::class, $category);
        $editForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($editForm->isSubmitted() && $editForm->isValid()) {
            // Zapis.
            $this->getDoctrine()->getManager()->flush();

            return $this->redirectToRoute('wi_admin_faq_category_show', [
                'id' => $category->getId(),
            ]);
        }

        return $this->render('WiAdminFAQBundle:Category:edit.html.php', [
            'category' => $category,
            'edit_form' => $editForm->createView(),
        ]);
    }

    /**
     * Deletes a category entity.
     *
     * @param Request $request
     * @param Category $category
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, Category $category)
    {
        // Utworzenie formularza usuwania kategorii.
        $formDelete = $this->createForm(CategoryDeleteType::class, $category);
        $formDelete->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($formDelete->isSubmitted() && $formDelete->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Usuwanie kategorii.
            $em->remove($category);
            $em->flush();

            return $this->redirectToRoute('wi_admin_faq_category_index');
        }

        return $this->render('WiAdminFAQBundle:Category:delete.html.php', [
            'category' => $category,
            'deleteForm' => $formDelete->createView(),
        ]);
    }

    /**
     * Update a order category entity.
     *
     * @param Category $category
     * @param string $sort_dir
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function updateOrderAction(Category $category, $sort_dir)
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $categoryRepo = $em->getRepository('WiAdminFAQBundle:Category');

        // Pobranie wszystkich aktywnych kategorii.
        $categories = $categoryRepo->findAllActive();

        // Poprawienie kolejności. Może wystąpić sytuacja, że kilka kategorii
        // będzie miała przypisany ten sam numer w kolejności i wtedy
        // przenoszenie o '1' nie spełnia swojej roli.
        $fixOrder = function ($categories, $order = 1) use (&$fixOrder, $em) {
            foreach ($categories as $category) {
                $category->setOrderNr($order++);

                $em->persist($category);
            }

            $em->flush();
        };

        $fixOrder($categories);

        switch ($sort_dir) {
            case 'up': // Przesunięcie w górę.
                $nextTo = $categoryRepo->findOneBy([
                    'orderNr' => $category->getOrderNr() - 1,
                ]);

                if (! is_null($nextTo)) {
                    $nextTo->setOrderNr($category->getOrderNr());
                    $category->setOrderNr($category->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            case 'down': // Przesunięcie w dół.
                $nextTo = $categoryRepo->findOneBy([
                    'orderNr' => $category->getOrderNr() + 1,
                ]);

                if (! is_null($nextTo)) {
                    $category->setOrderNr($nextTo->getOrderNr());
                    $nextTo->setOrderNr($nextTo->getOrderNr() - 1);

                    $em->persist($nextTo);
                }
                break;
            default:
                break;
        }

        $em->persist($category);
        $em->flush();

        return $this->redirectToRoute('wi_admin_faq_category_index');
    }
}
