<?php

namespace Wi\Admin\DownloadBundle\Controller;

use Knp\Component\Pager\Paginator;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\Filesystem\Exception\IOExceptionInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\CoreBundle\Form\DeleteType;
use Wi\Admin\DownloadBundle\Entity\File;
use Wi\Admin\DownloadBundle\Form\FileType;
use Wi\Admin\DownloadBundle\Utils\Upload;
use Wi\Admin\NewsBundle\Entity\News;

/**
 * File controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2018 WEBimpuls Sp. z o.o.
 */
class FileController extends Controller
{
    /**
     * Lista plików.
     *
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction(Request $request, Paginator $paginator)
    {
        $em = $this->getDoctrine()->getManager();
        $files = $paginator->paginate(
            $em->getRepository('WiAdminDownloadBundle:File')
                ->getFindAllActiveQuery(
                    $request->query->getInt('category_id'),
                    $request->query->get('q')
                ),
            $request->query->getInt('page', 1),
            $request->query->getInt('per_page') ?: 10
        );

        return $this->render('WiAdminDownloadBundle:File:index.html.php', [
            'files' => $files,
        ]);
    }

    /**
     * Dodawanie pliku.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request, Upload $uploader)
    {
        $file = new File();

        // Generowanie formularza.
        $form = $this->createForm(FileType::class, $file);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            $file->setExt($uploader->getExtension($file->getSystemName()));

            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();
            $news = $em->getRepository('WiAdminNewsBundle:News')
                ->find($request->query->getInt('news_id'))
            ;

            if ($news) {
                $file->setNews($news);
            }

            // Zapis.
            $em->persist($file);
            $em->flush();

            // Powrót do newsa, jeśli podane było jego ID.
            if ($news) {
                $route = 'wi_admin_news_show';

                if ($news->getType() == News::TYPE_BLOG) {
                    $route = 'wi_admin_blog_show';
                }

                return $this->redirectToRoute($route, [
                    'id' => $news->getId(),
                ]);
            }

            return $this->redirectToRoute('wi_admin_download_file_show', [
                'id' => $file->getId(),
            ]);
        }

        return $this->render('WiAdminDownloadBundle:File:new.html.php', [
            'file' => $file,
            'form' => $form->createView(),
        ]);
    }

    /**
     * Szczegóły pliku.
     *
     * @param File $file
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function showAction(File $file)
    {
        return $this->render('WiAdminDownloadBundle:File:show.html.php', [
            'file' => $file,
        ]);
    }

    /**
     * Edycja pliku.
     *
     * @param Request $request
     * @param File $file
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, File $file)
    {
        // Generowanie formularza.
        $form = $this->createForm(FileType::class, $file);
        $form->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();
            $news = $em->getRepository('WiAdminNewsBundle:News')
                ->find($request->query->getInt('news_id'))
            ;
            // Zapis.
            $em->persist($file);
            $em->flush();

            // Powrót do newsa, jeśli podane było jego ID.
            if ($news) {
                $route = 'wi_admin_news_show';

                if ($news->getType() == News::TYPE_BLOG) {
                    $route = 'wi_admin_blog_show';
                }

                return $this->redirectToRoute($route, [
                    'id' => $news->getId(),
                ]);
            }

            return $this->redirectToRoute('wi_admin_download_file_show', [
                'id' => $file->getId(),
            ]);
        }

        return $this->render('WiAdminDownloadBundle:File:edit.html.php', [
            'file' => $file,
            'form' => $form->createView(),
        ]);
    }

    /**
     * Usunięcie pliku.
     *
     * @param Request $request
     * @param File $file
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, File $file)
    {
        $form = $this->createForm(DeleteType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();
            $em->remove($file);
            $em->flush();

            return $this->redirectToRoute('wi_admin_download_file_index');
        }

        return $this->render('WiAdminDownloadBundle:File:delete.html.php', [
            'form' => $form->createView(),
            'file' => $file,
        ]);
    }

    public function uploadAction(Request $request, Upload $uploader)
    {
        // Sprawdzenie, czy istnieją pliki.
        if (empty($request->files) || $request->files->get('file')->getError()) {
            return new JsonResponse([
                'OK' => 0,
                'info' => 'Failed to move uploaded file.',
            ]);
        }

        $fs = new Filesystem();
        $dir = $this->getParameter('upload_dirs.files');

        // Sprawdzenie, czy katalog istnieje.
        if (! is_dir($dir)) {
            $fs->mkdir($dir);
        }

        // Ustawienie liczby chunków.
        $chunk = ! empty($request->request->get('chunk')) ? (int) $request->request->get('chunk') : 0;
        $chunks = ! empty($request->request->get('chunks')) ? (int) $request->request->get('chunks') : 0;

        // Ustawienie nazwy pliku.
        $fileName = ! empty($request->request->get('name')) ?
            strtolower($request->request->get('name')) :
            strtolower($request->files->get('file')->getClientOriginalName())
        ;

        // Ustawienie ścieżki do pliku.
        $filePath = $dir . $fileName;

        // Otworzenie pliku tymczasowego.
        $out = @fopen("{$filePath}.part", $chunk == 0 ? "wb" : "ab");

        if ($out) {
            // Read binary input stream and append it to temp file
            $in = @fopen($request->files->get('file')->getPathName(), "rb");

            if ($in) {
                while ($buff = fread($in, 4096)) {
                    fwrite($out, $buff);
                }
            } else {
                return new JsonResponse([
                    'OK' => 0,
                    'info' => 'Failed to open input stream.',
                ]);
            }

            @fclose($in);
            @fclose($out);

            @unlink($request->files->get('file')->getPathName());
        } else {
            return new JsonResponse([
                'OK' => 0,
                'info' => 'Failed to open output stream.',
            ]);
        }

        // Check if file has been uploaded
        if (! $chunks || $chunk == $chunks - 1) {
            // Prepare filename.
            $filename = $uploader->getEnsureFilenameUnique(
                str_replace(
                    basename($filePath),
                    '',
                    $filePath
                ),
                $uploader->prepareFilename(
                    basename($filePath),
                    preg_replace('/^.*\./', '', basename($filePath))
                )
            );

            // New file path.
            $newFilePath = str_replace(basename($filePath), '', $filePath) . $filename;

            // Strip the temp .part suffix off
            rename("{$filePath}.part", $newFilePath);

            return new JsonResponse([
                'OK' => 1,
                'info' => 'Upload successful.',
                'filename' => $filename,
            ]);
        }

        return new JsonResponse([
            'OK' => 1,
            'info' => 'Upload successful.',
        ]);
    }

    public function ajaxFileSaveAction(Request $request, Upload $uploader)
    {
        $rFile = $request->request->get('file');

        $file = new File();
        $file
            ->setSystemName($rFile['systemName'])
            ->setTitle($rFile['title'])
            ->setExt($uploader->getExtension($rFile['systemName']))
            ->setSize($rFile['size'])
        ;

        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $news = $em->getRepository('WiAdminNewsBundle:News')
            ->find($request->request->getInt('news_id'))
        ;

        if ($news) {
            $file->setNews($news);
        }

        // Zapis.
        $em->persist($file);
        $em->flush();

        return new JsonResponse([
            'id' => $file->getId(),
        ]);
    }
}
