<?php

namespace Wi\Admin\CoreBundle\Utils;

use claviska\SimpleImage;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Filesystem\Exception\IOExceptionInterface;
use Symfony\Component\Filesystem\Filesystem;

/**
 * Klasa do obsługi przesyłania pojedynczych plików.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class Upload
{
    /**
     * Allowed upload file extensions.
     *
     * @var array
     */
    protected $allowedExt = [
        'jpg',
        'jpeg',
        'png',
    ];

    /**
     * Array of possible sizes of images and their dimensions.
     *
     * @var array
     */
    protected $config = [];

    /**
     * Container Interface object.
     *
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var Slugger
     */
    private $slugger;

    /**
     * Directory to which to upload.
     *
     * @var string
     */
    protected $dir;

    /**
     * Construct.
     *
     * @param ContainerInterface $container
     */
    public function __construct(ContainerInterface $container)
    {
        // Set Container Interface variable.
        $this->container = $container;
        $this->slugger = $container->get('front.slugger');
    }

    /**
     * Format bytes to kilobytes, megabytes, gigabytes, terabytes.
     *
     * @param   int     $bytes
     * @return  string
     */
    public function formatBytes($bytes, $dec = 2)
    {
        $size = ['B', 'KB', 'MB', 'GB', 'TB'];
        $factor = floor((strlen($bytes) - 1) / 3);

        $number = $bytes / pow(1024, $factor);
        $number = ($dec == 0) ? ceil($number) : $number;

        return number_format($number, $dec) . ' ' .
            (isset($size[$factor]) ? $size[$factor] : null)
        ;
    }

    /**
     * Get image dimensions.
     *
     * @param   string      $filepath   Path to file.
     * @param   string      $ext        File extension.
     * @return  string|null
     */
    public function getDimensions($filepath, $ext)
    {
        $imgExt = ['jpg', 'jpeg', 'png', 'gif'];

        if (in_array(strtolower($ext), $imgExt) && file_exists($filepath)) {
            list($width, $height) = getimagesize($filepath);

            return $width . 'x' . $height . ' px';
        }

        return null;
    }

    /**
     * Get ensure dirname unique
     *
     * Checking dirname is unique. When dirname is not unique append to name
     * increment integer.
     *
     * @param   string  $dir        Directory where dir must be unique.
     * @param   string  $dirname    Dirname.
     * @return  string
     */
    public function getEnsureDirnameUnique($dir, $dirname)
    {
        $fs = new Filesystem();
        $new_dirname = $dirname;
        $i = 1;

        while ($fs->exists($dir.$new_dirname)) {
            $new_dirname = $dirname . '-' . $i;
            $i++;
        }

        return $new_dirname;
    }

    /**
     * Get ensure filename unique in directory
     *
     * Checking filename is unique in directory. When filename is not unique
     * append to name increment integer.
     *
     * @param   string  $dir        Directory where file must be unique.
     * @param   string  $filename   Filename.
     * @return  string
     */
    public function getEnsureFilenameUnique($dir, $filename)
    {
        $fs = new Filesystem();
        $fs->mkdir($dir);
        $ext = preg_replace('/^.*\./', '', $filename);
        $name = str_replace('.' . $ext, '', $filename);
        $new_name = $name;
        $i = 1;

        while ($fs->exists($dir.$new_name.'.'.$ext)) {
            $new_name = $name . '-' . $i;
            $i++;
        }

        return $new_name.'.'.$ext;
    }

    /**
     * Check file extension is allowed.
     *
     * @param   string  $filename
     * @return  bool
     */
    public function isAllowedExt($filename)
    {
        $ext = strtolower(preg_replace('/^.*\./', '', $filename));

        if (in_array($ext, $this->allowedExt)) {
            return true;
        }

        return false;
    }

    public function getExtension($filename)
    {
        return strtolower(preg_replace('/^.*\./', '', $filename));
    }

    /**
     * Prepare dirname.
     *
     * Convert special characters from dirname and remove not allowed chars,
     * whole convert to lower case.
     *
     * @param   string $dirname Directory name.
     * @return  string
     */
    public function prepareDirname($dirname)
    {
        return preg_replace(
            '/[\_|+ -]+/',
            '-',
            strtolower(
                preg_replace(
                    '/[^a-zA-Z0-9\_|+ -]/',
                    '',
                    iconv('UTF-8', 'ASCII//TRANSLIT', $dirname)
                )
            )
        );
    }

    /**
     * Prepare filename.
     *
     * Convert special characters from filename and remove not allowed chars,
     * filename cut to 200 chars whole convert to lower case.
     *
     * @param   string  $filename   Filename with extension.
     * @param   string  $ext        File extension.
     * @return  string
     */
    public function prepareFilename($filename, $ext)
    {
        $slugger = $this->container->get('front.slugger');

        return substr($slugger->slugify(str_replace('.'.$ext, '', $filename)), 0, 200) . '.' . strtolower($ext);
    }

    /**
     * Resize image and create thumbnails.
     *
     * @param   File    $file
     * @param   array   $params
     */
    public function process($file, $params)
    {
        $fs = new Filesystem();
        $img = new SimpleImage($file->getRealPath());

        // Resize and create thumbnails.
        foreach ($this->config as $suffix => $options) {
            $suffix = empty($suffix) ? '' : $suffix . DIRECTORY_SEPARATOR;
            $savePath = str_replace(
                $file->getFilename(),
                '',
                $file->getRealPath()
            ) . $suffix;

            // Create dir is not exists.
            $fs->mkdir($savePath);

            // Add filename to save path.
            $savePath .= $file->getFilename();

            // Crop main image to fixed dimensions.
            if (isset($params['cropSave']) && $params['cropSave'] && empty($suffix)) {
                $img
                    ->crop(
                        $params['crop_x'],
                        $params['crop_y'],
                        $params['crop_x'] + $params['crop_width'],
                        $params['crop_y'] + $params['crop_height']
                    )
                ;
            } else {
                $img->bestFit($options['width'], $options['height']);
            }

            $img->toFile($savePath);
        }
    }

    /**
     * Upload File.
     *
     * @param   UploadedFile    $file   Uploaded file.
     * @param   array           $params
     * @return  null|string
     */
    public function upload(UploadedFile $file, $params = [])
    {
        if (! in_array($file->getClientOriginalExtension(), $this->allowedExt)) {
            return null;
        }

        $filename = $this->getEnsureFilenameUnique(
            $this->dir,
            $this->prepareFilename(
                $file->getClientOriginalName(),
                $file->getClientOriginalExtension()
            )
        );
        $file = $file->move($this->dir, $filename);
        $this->process($file, $params);

        return $file->getFilename();
    }
}
