<?php

namespace Wi\Admin\CoreBundle\Service\Seo;

use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Routing\RouterInterface;
use Wi\Admin\CoreBundle\Service\Config;
use Wi\Admin\NewsBundle\Entity\Category;
use Wi\Admin\CoreBundle\Service\Utils;
use Wi\Admin\CoreBundle\Service\Seo\GenerateTypeInterface;
use Wi\Front\CoreBundle\Utils\Slugger;

/**
 * Seo generator for news category.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class SeoNewsCategory
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var Slugger
     */
    private $slugger;

    /**
     * @var Utils
     */
    private $utils;

    /**
     * Constructor.
     *
     * @param Config $config
     * @param RouterInterface $router
     * @param Slugger $slugger
     * @param Utils $utils
     */
    public function __construct(Config $config, RouterInterface $router, Slugger $slugger, Utils $utils)
    {
        $this->config = $config;
        $this->router = $router;
        $this->slugger = $slugger;
        $this->utils = $utils;
    }

    // -------------------------------------------------------------------------

    /**
     * Zwraca gotowe metadatane dla kategorii.
     *
     * @param Category $category
     * @return string
     */
    public function getMeta(Category $category)
    {
        $arr = [
            $this->metaTitleRender($category),
            $this->metaDescriptionRender($category),
            $this->metaKeywordsRender($category),
            $this->canonicalRender($category),
            $this->ogRender($category),
        ];

        return implode(PHP_EOL.'    ', $arr).PHP_EOL;
    }

    /**
     * Zwraca tablicę wygenerowanych metadanych dla kategorii wraz ze sposobem
     * generowania poszczególnych danych.
     *
     * @param Category $category
     * @return array
     */
    public function getMetaArray(Category $category)
    {
        $title = $this->getTitleGenerate($category);
        $desc = $this->getDescriptionGenerate($category);
        $keywords = $this->getKeywordsGenerate($category);
        $canonical = $this->getCanonicalGenerate($category);
        $image = $this->getImageGenerate($category);

        $this->getOgArray($category);

        return [
            'title' => [
                'type' => $title['type'],
                'value' => $title['value'],
                'length' => $this->getTitleLength($category),
            ],
            'description' => [
                'type' => $desc['type'],
                'value' => $desc['value'],
                'length' => $this->getDescriptionLength($category),
            ],
            'keywords' => [
                'type' => $keywords['type'],
                'value' => $keywords['value'],
            ],
            'canonical' => [
                'type' => $canonical['type'],
                'value' => $canonical['value'],
            ],
            'image' => [
                'type' => $image['type'],
                'value' => $image['value'],
            ],
        ];
    }

    /**
     * Zwraca tablicę metadanych dla kategorii.
     *
     * @param Category $category
     * @return array
     */
    public function getMetaArrayValues(Category $category)
    {
        $out = [];

        foreach ($this->getMetaArray($category) as $key => $value) {
            $out[$key] = $value['value'];
        }

        return $out;
    }

    /**
     * Zwraca tablicę wygenerowanych danych OpenGraph dla kategorii wraz ze sposobem
     * generowania poszczególnych danych.
     *
     * @param Category $category
     * @return array
     */
    public function getOgArray(Category $category)
    {
        $title = $this->getTitleGenerate($category);
        $desc = $this->getDescriptionGenerate($category);
        $image = $this->getImageGenerate($category);
        $type = $this->getOgTypeGenerate($category);

        return [
            'og_title' => [
                'type' => $title['type'],
                'value' => $title['value'],
            ],
            'og_description' => [
                'type' => $desc['type'],
                'value' => $desc['value'],
            ],
            'og_type' => [
                'type' => $type['type'],
                'value' => $type['value'],
            ],
            'og_image' => [
                'type' => $image['type'],
                'value' => $image['value'],
            ],
        ];
    }

    // -------------------------------------------------------------------------

    /**
     * Zwraca tablicę z wygenerowanym tytułem kategorii
     * oraz sposobem generowania.
     *
     * @param Category $category
     * @return array
     */
    public function getTitleGenerate(Category $category)
    {
        $titleForAllPages = boolval(intval($this->config->get('metadata.titleForAllPages')));
        $affix = $titleForAllPages ? $this->config->get('metadata.titleSeparator').$this->config->get('metadata.title') : '';

        if (! empty($category->getName())) {
            return [
                'type' => GenerateTypeInterface::DYNAMIC,
                'value' => htmlentities($category->getName().$affix),
            ];
        }

        return [
            'type' => GenerateTypeInterface::DEFAULT,
            'value' => htmlentities($this->config->get('metadata.title')),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym opisem kategorii
     * oraz sposobem generowania.
     *
     * @param Category $category
     * @return array
     */
    public function getDescriptionGenerate(Category $category)
    {
        if (! empty($category->getDescription())) {
            return [
                'type' => GenerateTypeInterface::DYNAMIC,
                'value' => htmlentities($category->getDescription()),
            ];
        }

        return [
            'type' => GenerateTypeInterface::DEFAULT,
            'value' => htmlentities($this->config->get('metadata.description')),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanymi słowami kluczowymi dla kategorii
     * oraz sposobem generowania.
     *
     * @param Category $category
     * @return array
     */
    public function getKeywordsGenerate(Category $category)
    {
        $keywordsForAllPages = boolval(intval($this->config->get('metadata.keywordsForAllPages')));

        if (boolval(intval($this->config->get('metadata.keywordsEnable')))) {
            if ($keywordsForAllPages) {
                return [
                    'type' => GenerateTypeInterface::DEFAULT,
                    'value' => $this->config->get('metadata.keywords'),
                ];
            }
        }

        return [
            'type' => GenerateTypeInterface::EMPTY,
            'value' => null,
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym kanonicznym adresem URL dla kategorii
     * oraz sposobem generowania.
     *
     * @param Category $category
     * @return array
     */
    public function getCanonicalGenerate(Category $category)
    {
        return [
            'type' => GenerateTypeInterface::DYNAMIC,
            'value' => $this->router->generate(
                'wi_front_news_by_category_index',
                [
                    'id' => $category->getId(),
                    'slug' => (! empty($category->getName()) ? $this->slugger->slugify($category->getName()) : '-'),
                ],
                UrlGeneratorInterface::ABSOLUTE_URL
            ),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym zdjęciem OpenGraph kategorii oraz
     * sposobem generowania.
     *
     * @param Category $category
     * @return array
     */
    public function getImageGenerate(Category $category)
    {
        $openGraphEnable = boolval(intval($this->config->get('metadata.openGraphEnable')));

        if ($openGraphEnable) {
            if (! empty($img = $this->config->get('metadata.openGraphDefaultImage'))) {
                return [
                    'type' => GenerateTypeInterface::DEFAULT,
                    'value' => $img,
                ];
            }
        }

        return [
            'type' => GenerateTypeInterface::EMPTY,
            'value' => null,
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym typem OpenGraph dla strony oraz
     * sposobem generowania.
     *
     * @param Category $category
     * @return array
     */
    public function getOgTypeGenerate(Category $category)
    {
        return [
            'type' => GenerateTypeInterface::DYNAMIC,
            'value' => 'news',
        ];
    }

    // -------------------------------------------------------------------------



    // -------------------------------------------------------------------------

    /**
     * Sprawdza, czy tytuł jest właściwej długości.
     *
     * @param Category $category
     * @return array
     */
    public function getTitleLength(Category $category)
    {
        $length = mb_strlen(html_entity_decode($this->getTitleGenerate($category)['value']));
        $from = intval($this->config->get('seo.metaTitleLengthFrom'));
        $to = intval($this->config->get('seo.metaTitleLengthTo'));

        if ($length > $to) {
            $status = LengthStatusInterface::TOO_LONG;
        } elseif ($length >= $from && $length <= $to) {
            $status = LengthStatusInterface::RIGHT;
        } else {
            $status = LengthStatusInterface::TOO_SHORT;
        }

        return [
            'length' => $length,
            'status' => $status,
        ];
    }

    /**
     * Sprawdza, czy opis jest właściwej długości.
     *
     * @param Category $category
     * @return array
     */
    public function getDescriptionLength(Category $category)
    {
        $length = mb_strlen(html_entity_decode($this->getDescriptionGenerate($category)['value']));
        $from = intval($this->config->get('seo.metaDescriptionLengthFrom'));
        $to = intval($this->config->get('seo.metaDescriptionLengthTo'));

        if ($length > $to) {
            $status = LengthStatusInterface::TOO_LONG;
        } elseif ($length >= $from && $length <= $to) {
            $status = LengthStatusInterface::RIGHT;
        } else {
            $status = LengthStatusInterface::TOO_SHORT;
        }

        return [
            'length' => $length,
            'status' => $status,
        ];
    }

    // -------------------------------------------------------------------------

    /**
     * Renderuje tytuł strony.
     *
     * @param Category $category
     * @return string
     */
    public function metaTitleRender(Category $category)
    {
        return sprintf('<title>%s</title>', $this->getTitleGenerate($category)['value']);
    }

    /**
     * Renderuje opis strony.
     *
     * @param Category $category
     * @return string
     */
    public function metaDescriptionRender(Category $category)
    {
        return sprintf(
            '<meta name="description" content="%s">',
            $this->getDescriptionGenerate($category)['value']
        );
    }

    /**
     * Renderuje słowa kluczowe strony.
     *
     * @param Category $category
     * @return string
     */
    public function metaKeywordsRender(Category $category)
    {
        if (boolval(intval($this->config->get('metadata.keywordsEnable')))) {
            return sprintf(
                '<meta name="keywords" content="%s">',
                $this->getKeywordsGenerate($category)['value']
            );
        }

        return;
    }

    /**
     * Renderuje kanoniczny adres URL.
     *
     * @param Category $category
     * @return string
     */
    public function canonicalRender(Category $category)
    {
        if (boolval(intval($this->config->get('metadata.canonical')))) {
            return sprintf(
                '<link rel="canonical" href="%s">',
                $this->getCanonicalGenerate($category)['value']
            );
        }

        return;
    }

    // -------------------------------------------------------------------------

    /**
     * Renderuje OpenGraph.
     *
     * @param Category $category
     * @return string
     */
    public function ogRender(Category $category)
    {
        if (boolval(intval($this->config->get('metadata.openGraphEnable')))) {
            $arr = [
                $this->ogSiteNameRender($category),
                $this->ogTitleRender($category),
                $this->ogDescriptionRender($category),
                $this->ogTypeRender($category),
                $this->ogUrlRender($category),
                $this->ogImageRender($category),
            ];

            return trim(implode(PHP_EOL.'    ', $arr).PHP_EOL);
        }

        return;
    }

    /**
     * Renderuje nazwę strony dla OpenGraph.
     *
     * @param Category $category
     * @return string
     */
    public function ogSiteNameRender(Category $category)
    {
        return sprintf(
            '<meta property="og:site_name" content="%s">',
            $this->getTitleGenerate($category)['value']
        );
    }

    /**
     * Renderuje tytuł dla OpenGraph.
     *
     * @param Category $category
     * @return string
     */
    public function ogTitleRender(Category $category)
    {
        return sprintf(
            '<meta property="og:title" content="%s">',
            $this->getTitleGenerate($category)['value']
        );
    }

    /**
     * Renderuje opis strony dla OpenGraph.
     *
     * @param Category $category
     * @return string
     */
    public function ogDescriptionRender(Category $category)
    {
        return sprintf(
            '<meta property="og:description" content="%s">',
            $this->getDescriptionGenerate($category)['value']
        );
    }

    /**
     * Renderuje typ treści dla OpenGraph.
     *
     * @param Category $category
     * @return string
     */
    public function ogTypeRender(Category $category)
    {
        return sprintf(
            '<meta property="og:type" content="%s">',
            $this->getOgTypeGenerate($category)['value']
        );
    }

    /**
     * Renderuje adres URL dla OpenGraph.
     *
     * @param Category $category
     * @return string
     */
    public function ogUrlRender(Category $category)
    {
        return sprintf(
            '<meta property="og:url" content="%s">',
            $this->getCanonicalGenerate($category)['value']
        );
    }

    /**
     * Renderuje zdjęcie dla OpenGraph.
     *
     * @param Category $category
     * @return string
     */
    public function ogImageRender(Category $category)
    {
        if (! empty($img = $this->getImageGenerate($category)['value'])) {
            return sprintf(
                '<meta property="og:image" content="%s">',
                $img
            );
        }

        if (! empty($img = $this->config->get('metadata.openGraphDefaultImage'))) {
            return sprintf(
                '<meta property="og:image" content="%s">',
                $img
            );
        }

        return;
    }
}
