<?php

namespace Wi\Admin\CoreBundle\Service\Seo;

use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Routing\RouterInterface;
use Wi\Admin\CoreBundle\Service\Config;
use Wi\Admin\NewsBundle\Entity\News;
use Wi\Admin\CoreBundle\Service\Utils;
use Wi\Admin\CoreBundle\Service\Seo\GenerateTypeInterface;
use Wi\Front\CoreBundle\Utils\Slugger;

/**
 * Seo generator for news.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class SeoNews
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var Slugger
     */
    private $slugger;

    /**
     * @var Utils
     */
    private $utils;

    /**
     * Constructor.
     *
     * @param Config $config
     * @param RouterInterface $router
     * @param Slugger $slugger
     * @param Utils $utils
     */
    public function __construct(Config $config, RouterInterface $router, Slugger $slugger, Utils $utils)
    {
        $this->config = $config;
        $this->router = $router;
        $this->slugger = $slugger;
        $this->utils = $utils;
    }

    // -------------------------------------------------------------------------

    /**
     * Zwraca gotowe metadatane dla aktualności.
     *
     * @param News $news
     * @return string
     */
    public function getMeta(News $news)
    {
        $arr = [
            $this->metaTitleRender($news),
            $this->metaDescriptionRender($news),
            $this->metaKeywordsRender($news),
            $this->canonicalRender($news),
            $this->ogRender($news),
        ];

        return implode(PHP_EOL.'    ', $arr).PHP_EOL;
    }

    /**
     * Zwraca tablicę wygenerowanych metadanych dla aktualności wraz ze sposobem
     * generowania poszczególnych danych.
     *
     * @param News $news
     * @return array
     */
    public function getMetaArray(News $news)
    {
        $title = $this->getTitleGenerate($news);
        $desc = $this->getDescriptionGenerate($news);
        $keywords = $this->getKeywordsGenerate($news);
        $canonical = $this->getCanonicalGenerate($news);
        $image = $this->getImageGenerate($news);

        return [
            'title' => [
                'type' => $title['type'],
                'value' => $title['value'],
                'length' => $this->getTitleLength($news),
            ],
            'description' => [
                'type' => $desc['type'],
                'value' => $desc['value'],
                'length' => $this->getDescriptionLength($news),
            ],
            'keywords' => [
                'type' => $keywords['type'],
                'value' => $keywords['value'],
            ],
            'canonical' => [
                'type' => $canonical['type'],
                'value' => $canonical['value'],
            ],
            'image' => [
                'type' => $image['type'],
                'value' => $image['value'],
            ],
        ];
    }

    /**
     * Zwraca tablicę metadanych dla aktualności.
     *
     * @param News $news
     * @return array
     */
    public function getMetaArrayValues(News $news)
    {
        $out = [];

        foreach ($this->getMetaArray($news) as $key => $value) {
            $out[$key] = $value['value'];
        }

        return $out;
    }

    /**
     * Zwraca tablicę wygenerowanych danych OpenGraph dla aktualności
     * wraz ze sposobem generowania poszczególnych danych.
     *
     * @param News $news
     * @return array
     */
    public function getOgArray(News $news)
    {
        $title = $this->getTitleGenerate($news);
        $desc = $this->getDescriptionGenerate($news);
        $image = $this->getImageGenerate($news);
        $type = $this->getOgTypeGenerate($news);

        return [
            'og_title' => [
                'type' => $title['type'],
                'value' => $title['value'],
            ],
            'og_description' => [
                'type' => $desc['type'],
                'value' => $desc['value'],
            ],
            'og_type' => [
                'type' => $type['type'],
                'value' => $type['value'],
            ],
            'og_image' => [
                'type' => $image['type'],
                'value' => $image['value'],
            ],
        ];
    }

    // -------------------------------------------------------------------------

    /**
     * Zwraca tablicę z wygenerowanym kanonicznym adresem URL dla aktualności
     * oraz sposobem generowania.
     *
     * @param News $news
     * @return array
     */
    public function getCanonicalGenerate(News $news)
    {
        if (boolval(intval($this->config->get('metadata.canonical')))) {
            if (! empty($news->getMetaCanonical())) {
                if (! boolval(intval($this->config->get('metadata.canonicalForce')))) {
                    return [
                        'type' => GenerateTypeInterface::MANUAL,
                        'value' => $news->getMetaCanonical(),
                    ];
                }
            }
        }

        return [
            'type' => GenerateTypeInterface::DYNAMIC,
            'value' => $this->router->generate(
                'wi_front_news_show',
                [
                    'id' => $news->getId(),
                    'slug' => (! empty($news->getTitle()) ? $this->slugger->slugify($news->getTitle()) : '-'),
                ],
                UrlGeneratorInterface::ABSOLUTE_URL
            ),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym opisem aktualności
     * oraz sposobem generowania.
     *
     * @param News $news
     * @return array
     */
    public function getDescriptionGenerate(News $news)
    {
        if (! empty($news->getMetaDescription())) {
            return [
                'type' => GenerateTypeInterface::MANUAL,
                'value' => htmlentities($news->getMetaDescription()),
            ];
        }

        if (! empty($news->getShortcut())) {
            return [
                'type' => GenerateTypeInterface::DYNAMIC,
                'value' => htmlentities($news->getShortcut()),
            ];
        }

        return [
            'type' => GenerateTypeInterface::DEFAULT,
            'value' => htmlentities($this->config->get('metadata.description')),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym zdjęciem OpenGraph aktualności
     * oraz sposobem generowania.
     *
     * @param News $news
     * @return array
     */
    public function getImageGenerate(News $news)
    {
        $openGraphEnable = boolval(intval($this->config->get('metadata.openGraphEnable')));

        if ($openGraphEnable) {
            if (! empty($news->getMetaImage())) {
                return [
                    'type' => GenerateTypeInterface::MANUAL,
                    'value' => $news->getMetaImage(),
                ];
            }

            if (! empty($image = $news->getImage())) {
                return [
                    'type' => GenerateTypeInterface::DYNAMIC,
                    'value' => $this->utils->absoluteUrl('media/assets/News/thumbnail/'.$image),
                ];
            }

            if (! empty($img = $this->config->get('metadata.openGraphDefaultImage'))) {
                return [
                    'type' => GenerateTypeInterface::DEFAULT,
                    'value' => $img,
                ];
            }
        }

        return [
            'type' => GenerateTypeInterface::EMPTY,
            'value' => null,
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanymi słowami kluczowymi dla aktualności
     * oraz sposobem generowania.
     *
     * @param News $news
     * @return array
     */
    public function getKeywordsGenerate(News $news)
    {
        $keywordsForAllPages = boolval(intval($this->config->get('metadata.keywordsForAllPages')));
        $affix = ! empty($news->getMetaKeywords()) ? ', ' : '';
        $affix .= $keywordsForAllPages ? $this->config->get('metadata.keywords') : '';

        if (boolval(intval($this->config->get('metadata.keywordsEnable')))) {
            if (! empty($news->getMetaKeywords())) {
                return [
                    'type' => GenerateTypeInterface::MANUAL,
                    'value' => $news->getMetaKeywords().$affix,
                ];
            }

            return [
                'type' => GenerateTypeInterface::DEFAULT,
                'value' => $news->getMetaKeywords().$affix,
            ];
        }

        return [
            'type' => GenerateTypeInterface::EMPTY,
            'value' => null,
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym typem OpenGraph dla aktualności
     * oraz sposobem generowania.
     *
     * @return array
     */
    public function getOgTypeGenerate(News $news)
    {
        return [
            'type' => GenerateTypeInterface::DYNAMIC,
            'value' => $this->config->get('og.newsType'),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym tytułem aktualności
     * oraz sposobem generowania.
     *
     * @param News $news
     * @return array
     */
    public function getTitleGenerate(News $news)
    {
        $titleForAllPages = boolval(intval($this->config->get('metadata.titleForAllPages')));
        $affix = $titleForAllPages ? $this->config->get('metadata.titleSeparator').$this->config->get('metadata.title') : '';

        if (! empty($news->getMetaTitle())) {
            return [
                'type' => GenerateTypeInterface::MANUAL,
                'value' => htmlentities($news->getMetaTitle().$affix),
            ];
        }

        if (! empty($news->getTitle())) {
            return [
                'type' => GenerateTypeInterface::DYNAMIC,
                'value' => htmlentities($news->getTitle().$affix, ENT_QUOTES),
            ];
        }

        return [
            'type' => GenerateTypeInterface::DEFAULT,
            'value' => htmlentities($this->config->get('metadata.title')),
        ];
    }

    // -------------------------------------------------------------------------

    /**
     * Renderuje tytuł aktualności.
     *
     * @param News $news
     * @return string
     */
    public function metaTitleRender(News $news)
    {
        return sprintf('<title>%s</title>', $this->getTitleGenerate($news)['value']);
    }

    /**
     * Renderuje opis aktualności.
     *
     * @param News $news
     * @return string
     */
    public function metaDescriptionRender(News $news)
    {
        return sprintf(
            '<meta name="description" content="%s">',
            $this->getDescriptionGenerate($news)['value']
        );
    }

    /**
     * Renderuje słowa kluczowe aktualności.
     *
     * @param News $news
     * @return string
     */
    public function metaKeywordsRender(News $news)
    {
        if (boolval(intval($this->config->get('metadata.keywordsEnable')))) {
            return sprintf(
                '<meta name="keywords" content="%s">',
                $this->getKeywordsGenerate($news)['value']
            );
        }

        return;
    }

    /**
     * Renderuje kanoniczny adres URL.
     *
     * @param News $news
     * @return string
     */
    public function canonicalRender(News $news)
    {
        if (boolval(intval($this->config->get('metadata.canonical')))) {
            return sprintf(
                '<link rel="canonical" href="%s">',
                $this->getCanonicalGenerate($news)['value']
            );
        }

        return;
    }

    // -------------------------------------------------------------------------

    /**
     * Renderuje OpenGraph.
     *
     * @param News $news
     * @return string
     */
    public function ogRender(News $news)
    {
        if (boolval(intval($this->config->get('metadata.openGraphEnable')))) {
            $arr = [
                $this->ogSiteNameRender($news),
                $this->ogTitleRender($news),
                $this->ogDescriptionRender($news),
                $this->ogTypeRender($news),
                $this->ogUrlRender($news),
                $this->ogImageRender($news),
            ];

            return trim(implode(PHP_EOL.'    ', $arr).PHP_EOL);
        }

        return;
    }

    /**
     * Renderuje nazwę aktualności dla OpenGraph.
     *
     * @param News $news
     * @return string
     */
    public function ogSiteNameRender(News $news)
    {
        return sprintf(
            '<meta property="og:site_name" content="%s">',
            $this->getTitleGenerate($news)['value']
        );
    }

    /**
     * Renderuje tytuł dla OpenGraph.
     *
     * @param News $news
     * @return string
     */
    public function ogTitleRender(News $news)
    {
        return sprintf(
            '<meta property="og:title" content="%s">',
            $this->getTitleGenerate($news)['value']
        );
    }

    /**
     * Renderuje opis aktualności dla OpenGraph.
     *
     * @param News $news
     * @return string
     */
    public function ogDescriptionRender(News $news)
    {
        return sprintf(
            '<meta property="og:description" content="%s">',
            $this->getDescriptionGenerate($news)['value']
        );
    }

    /**
     * Renderuje typ treści dla OpenGraph.
     *
     * @param News $news
     * @return string
     */
    public function ogTypeRender(News $news)
    {
        return sprintf(
            '<meta property="og:type" content="%s">',
            $this->getOgTypeGenerate($news)['value']
        );
    }

    /**
     * Renderuje adres URL dla OpenGraph.
     *
     * @param News $news
     * @return string
     */
    public function ogUrlRender(News $news)
    {
        return sprintf(
            '<meta property="og:url" content="%s">',
            $this->getCanonicalGenerate($news)['value']
        );
    }

    /**
     * Renderuje zdjęcie dla OpenGraph.
     *
     * @param News $news
     * @return string
     */
    public function ogImageRender(News $news)
    {
        if (! empty($img = $this->getImageGenerate($news)['value'])) {
            return sprintf(
                '<meta property="og:image" content="%s">',
                $img
            );
        }

        if (! empty($img = $this->config->get('metadata.openGraphDefaultImage'))) {
            return sprintf(
                '<meta property="og:image" content="%s">',
                $img
            );
        }

        return;
    }

    // -------------------------------------------------------------------------

    /**
     * Sprawdza, czy tytuł jest właściwej długości.
     *
     * @param News $news
     * @return array
     */
    public function getTitleLength(News $news)
    {
        $length = mb_strlen(html_entity_decode($this->getTitleGenerate($news)['value']));
        $from = intval($this->config->get('seo.metaTitleLengthFrom'));
        $to = intval($this->config->get('seo.metaTitleLengthTo'));

        if ($length > $to) {
            $status = LengthStatusInterface::TOO_LONG;
        } elseif ($length >= $from && $length <= $to) {
            $status = LengthStatusInterface::RIGHT;
        } else {
            $status = LengthStatusInterface::TOO_SHORT;
        }

        return [
            'length' => $length,
            'status' => $status,
        ];
    }

    /**
     * Sprawdza, czy opis jest właściwej długości.
     *
     * @param News $news
     * @return array
     */
    public function getDescriptionLength(News $news)
    {
        $length = mb_strlen(html_entity_decode($this->getDescriptionGenerate($news)['value']));
        $from = intval($this->config->get('seo.metaDescriptionLengthFrom'));
        $to = intval($this->config->get('seo.metaDescriptionLengthTo'));

        if ($length > $to) {
            $status = LengthStatusInterface::TOO_LONG;
        } elseif ($length >= $from && $length <= $to) {
            $status = LengthStatusInterface::RIGHT;
        } else {
            $status = LengthStatusInterface::TOO_SHORT;
        }

        return [
            'length' => $length,
            'status' => $status,
        ];
    }
}
