<?php

namespace Wi\Admin\CoreBundle\Service\Seo;

use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Routing\RouterInterface;
use Wi\Admin\ContactBundle\Entity\Form;
use Wi\Admin\CoreBundle\Service\Config;
use Wi\Admin\CoreBundle\Service\Seo\GenerateTypeInterface;
use Wi\Admin\CoreBundle\Service\Seo\LengthStatusInterface;
use Wi\Admin\CoreBundle\Service\Seo\SeoPage;
use Wi\Front\CoreBundle\Utils\Slugger;

/**
 * Seo form.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class SeoForm
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var Slugger
     */
    private $slugger;

    /**
     * @var SeoPage
     */
    private $seoPage;

    /**
     * Constructor.
     *
     * @param Config $config
     * @param RouterInterface $router
     * @param Slugger $slugger
     */
    public function __construct(Config $config, RouterInterface $router, SeoPage $seoPage, Slugger $slugger)
    {
        $this->config = $config;
        $this->router = $router;
        $this->seoPage = $seoPage;
        $this->slugger = $slugger;
    }

    // -------------------------------------------------------------------------

    /**
     * Zwraca gotowe metadatane dla strony.
     *
     * @param Form $form
     * @return string
     */
    public function getMeta(Form $form)
    {
        $arr = [
            $this->metaTitleRender($form),
            $this->metaDescriptionRender($form),
            $this->metaKeywordsRender($form),
            $this->canonicalRender($form),
            $this->ogRender($form),
        ];

        return implode(PHP_EOL.'    ', $arr).PHP_EOL;
    }

    /**
     * Zwraca tablicę wygenerowanych metadanych wraz ze sposobem generowania
     * poszczególnych danych.
     *
     * @param Form $form
     * @return array
     */
    public function getMetaArray(Form $form)
    {
        $title = $this->getTitleGenerate($form);
        $desc = $this->getDescriptionGenerate($form);
        $keywords = $this->getKeywordsGenerate($form);
        $canonical = $this->getCanonicalGenerate($form);
        $image = $this->getImageGenerate($form);

        return [
            'title' => [
                'type' => $title['type'],
                'value' => $title['value'],
                'length' => $this->getTitleLength($form),
            ],
            'description' => [
                'type' => $desc['type'],
                'value' => $desc['value'],
                'length' => $this->getDescriptionLength($form),
            ],
            'keywords' => [
                'type' => $keywords['type'],
                'value' => $keywords['value'],
            ],
            'canonical' => [
                'type' => $canonical['type'],
                'value' => $canonical['value'],
            ],
            'image' => [
                'type' => $image['type'],
                'value' => $image['value'],
            ],
        ];
    }

    /**
     * Zwraca tablicę metadanych.
     *
     * @param Form $form
     * @return array
     */
    public function getMetaArrayValues(Form $form)
    {
        $out = [];

        foreach ($this->getMetaArray($form) as $key => $value) {
            $out[$key] = $value['value'];
        }

        return $out;
    }

    /**
     * Zwraca tablicę wygenerowanych danych OpenGraph dla wraz ze sposobem
     * generowania poszczególnych danych.
     *
     * @param Form $form
     * @return array
     */
    public function getOgArray(Form $form)
    {
        $title = $this->getTitleGenerate($form);
        $desc = $this->getDescriptionGenerate($form);
        $image = $this->getImageGenerate($form);
        $type = $this->getOgTypeGenerate($form);

        return [
            'og_title' => [
                'type' => $title['type'],
                'value' => $title['value'],
            ],
            'og_description' => [
                'type' => $desc['type'],
                'value' => $desc['value'],
            ],
            'og_type' => [
                'type' => $type['type'],
                'value' => $type['value'],
            ],
            'og_image' => [
                'type' => $image['type'],
                'value' => $image['value'],
            ],
        ];
    }

    // -------------------------------------------------------------------------

    /**
     * Zwraca tablicę z wygenerowanym tytułem oraz sposobem generowania.
     *
     * @param Form $form
     * @return array
     */
    public function getTitleGenerate(Form $form)
    {
        if (! is_null($form->getPage())) {
            $page = $form->getPage();

            if (is_null($page->getDateDeleted())) {
                return $this->seoPage->getTitleGenerate($page);
            }
        }

        return [
            'type' => GenerateTypeInterface::DEFAULT,
            'value' => htmlentities($this->config->get('metadata.title')),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym opisem oraz sposobem generowania.
     *
     * @param Form $form
     * @return array
     */
    public function getDescriptionGenerate(Form $form)
    {
        if (! is_null($form->getPage())) {
            $page = $form->getPage();

            if (is_null($page->getDateDeleted())) {
                return $this->seoPage->getDescriptionGenerate($page);
            }
        }

        return [
            'type' => GenerateTypeInterface::DEFAULT,
            'value' => htmlentities($this->config->get('metadata.description')),
        ];
    }

    /**
     * Zwraa tablicę z wygenerowanymi słowami kluczowymi oraz
     * sposobem generowania.
     *
     * @param Form $form
     * @return array
     */
    public function getKeywordsGenerate(Form $form)
    {
        if (! is_null($form->getPage())) {
            $page = $form->getPage();

            if (is_null($page->getDateDeleted())) {
                return $this->seoPage->getKeywordsGenerate($page);
            }
        }

        return [
            'type' => GenerateTypeInterface::EMPTY,
            'value' => null,
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym kanonicznym adresem URL oraz
     * sposobem generowania.
     *
     * @param Form $form
     * @return array
     */
    public function getCanonicalGenerate(Form $form)
    {
        if (boolval(intval($this->config->get('metadata.canonical')))) {
            if (! is_null($form->getPage())) {
                $page = $form->getPage();

                if (is_null($page->getDateDeleted())) {
                    if (! empty($page->getMetaCanonical())) {
                        if (! boolval(intval($this->config->get('metadata.canonicalForce')))) {
                            return [
                                'type' => GenerateTypeInterface::MANUAL,
                                'value' => $page->getMetaCanonical(),
                            ];
                        }
                    }
                }
            }
        }

        return [
            'type' => GenerateTypeInterface::DYNAMIC,
            'value' => $this->router->generate(
                'wi_front_contact_index_2',
                [
                    'id' => $form->getId(),
                    'slug' => (! empty($form->getName()) ? $this->slugger->slugify($form->getName()) : '-'),
                ],
                UrlGeneratorInterface::ABSOLUTE_URL
            ),
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym zdjęciem OpenGraph oraz
     * sposobem generowania.
     *
     * @param Form $form
     * @return array
     */
    public function getImageGenerate(Form $form)
    {
        if (! is_null($form->getPage())) {
            $page = $form->getPage();

            if (is_null($page->getDateDeleted())) {
                return $this->seoPage->getImageGenerate($page);
            }
        }

        if (! empty($img = $this->config->get('metadata.openGraphDefaultImage'))) {
            return [
                'type' => GenerateTypeInterface::DEFAULT,
                'value' => $img,
            ];
        }

        return [
            'type' => GenerateTypeInterface::EMPTY,
            'value' => null,
        ];
    }

    /**
     * Zwraca tablicę z wygenerowanym typem OpenGraph dla oraz
     * sposobem generowania.
     *
     * @param Form $form
     * @return array
     */
    public function getOgTypeGenerate(Form $form)
    {
        return [
            'type' => GenerateTypeInterface::DYNAMIC,
            'value' => $this->config->get('og.pageType'),
        ];
    }

    // -------------------------------------------------------------------------

    /**
     * Renderuje tytuł strony.
     *
     * @param Form $form
     * @return string
     */
    public function metaTitleRender(Form $form)
    {
        return sprintf('<title>%s</title>', $this->getTitleGenerate($form)['value']);
    }

    /**
     * Renderuje opis strony.
     *
     * @param Form $form
     * @return string
     */
    public function metaDescriptionRender(Form $form)
    {
        return sprintf(
            '<meta name="description" content="%s">',
            $this->getDescriptionGenerate($form)['value']
        );
    }

    /**
     * Renderuje słowa kluczowe strony.
     *
     * @param Form $form
     * @return string
     */
    public function metaKeywordsRender(Form $form)
    {
        if (boolval(intval($this->config->get('metadata.keywordsEnable')))) {
            return sprintf(
                '<meta name="keywords" content="%s">',
                $this->getKeywordsGenerate($form)['value']
            );
        }

        return;
    }

    /**
     * Renderuje kanoniczny adres URL.
     *
     * @param Form $form
     * @return string
     */
    public function canonicalRender(Form $form)
    {
        if (boolval(intval($this->config->get('metadata.canonical')))) {
            return sprintf(
                '<link rel="canonical" href="%s">',
                $this->getCanonicalGenerate($form)['value']
            );
        }

        return;
    }

    // -------------------------------------------------------------------------

    /**
     * Renderuje OpenGraph.
     *
     * @param Form $form
     * @return string
     */
    public function ogRender(Form $form)
    {
        if (boolval(intval($this->config->get('metadata.openGraphEnable')))) {
            $arr = [
                $this->ogSiteNameRender($form),
                $this->ogTitleRender($form),
                $this->ogDescriptionRender($form),
                $this->ogTypeRender($form),
                $this->ogUrlRender($form),
                $this->ogImageRender($form),
            ];

            return trim(implode(PHP_EOL.'    ', $arr).PHP_EOL);
        }

        return;
    }

    /**
     * Renderuje nazwę strony dla OpenGraph.
     *
     * @param Form $form
     * @return string
     */
    public function ogSiteNameRender(Form $form)
    {
        return sprintf(
            '<meta property="og:site_name" content="%s">',
            $this->getTitleGenerate($form)['value']
        );
    }

    /**
     * Renderuje tytuł dla OpenGraph.
     *
     * @param Form $form
     * @return string
     */
    public function ogTitleRender(Form $form)
    {
        return sprintf(
            '<meta property="og:title" content="%s">',
            $this->getTitleGenerate($form)['value']
        );
    }

    /**
     * Renderuje opis strony dla OpenGraph.
     *
     * @param Form $form
     * @return string
     */
    public function ogDescriptionRender(Form $form)
    {
        return sprintf(
            '<meta property="og:description" content="%s">',
            $this->getDescriptionGenerate($form)['value']
        );
    }

    /**
     * Renderuje typ treści dla OpenGraph.
     *
     * @param Form $form
     * @return string
     */
    public function ogTypeRender(Form $form)
    {
        return sprintf(
            '<meta property="og:type" content="%s">',
            $this->getOgTypeGenerate($form)['value']
        );
    }

    /**
     * Renderuje adres URL dla OpenGraph.
     *
     * @param Form $form
     * @return string
     */
    public function ogUrlRender(Form $form)
    {
        return sprintf(
            '<meta property="og:url" content="%s">',
            $this->getCanonicalGenerate($form)['value']
        );
    }

    /**
     * Renderuje zdjęcie dla OpenGraph.
     *
     * @param Form $form
     * @return string
     */
    public function ogImageRender(Form $form)
    {
        if (! empty($img = $this->getImageGenerate($form)['value'])) {
            return sprintf(
                '<meta property="og:image" content="%s">',
                $img
            );
        }

        if (! empty($img = $this->config->get('metadata.openGraphDefaultImage'))) {
            return sprintf(
                '<meta property="og:image" content="%s">',
                $img
            );
        }

        return;
    }

    // -------------------------------------------------------------------------

    /**
     * Sprawdza, czy tytuł jest właściwej długości.
     *
     * @param Form $form
     * @return array
     */
    public function getTitleLength(Form $form)
    {
        $length = mb_strlen(html_entity_decode($this->getTitleGenerate($form)['value']));
        $from = intval($this->config->get('seo.metaTitleLengthFrom'));
        $to = intval($this->config->get('seo.metaTitleLengthTo'));

        if ($length > $to) {
            $status = LengthStatusInterface::TOO_LONG;
        } elseif ($length >= $from && $length <= $to) {
            $status = LengthStatusInterface::RIGHT;
        } else {
            $status = LengthStatusInterface::TOO_SHORT;
        }

        return [
            'length' => $length,
            'status' => $status,
        ];
    }

    /**
     * Sprawdza, czy opis jest właściwej długości.
     *
     * @param Form $form
     * @return array
     */
    public function getDescriptionLength(Form $form)
    {
        $length = mb_strlen(html_entity_decode($this->getDescriptionGenerate($form)['value']));
        $from = intval($this->config->get('seo.metaDescriptionLengthFrom'));
        $to = intval($this->config->get('seo.metaDescriptionLengthTo'));

        if ($length > $to) {
            $status = LengthStatusInterface::TOO_LONG;
        } elseif ($length >= $from && $length <= $to) {
            $status = LengthStatusInterface::RIGHT;
        } else {
            $status = LengthStatusInterface::TOO_SHORT;
        }

        return [
            'length' => $length,
            'status' => $status,
        ];
    }
}
