<?php

namespace Wi\Admin\CoreBundle\Service\Seo;

use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Routing\RouterInterface;
use Wi\Admin\CoreBundle\Service\Config;
use Wi\Front\CoreBundle\Utils\Slugger;

/**
 * Seo default.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class SeoDefault
{
    /**
     * @var Config
     */
    private $config;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var Slugger
     */
    private $slugger;

    /**
     * Constructor.
     *
     * @param Config $config
     * @param RouterInterface $router
     * @param Slugger $slugger
     */
    public function __construct(Config $config, RouterInterface $router, Slugger $slugger)
    {
        $this->config = $config;
        $this->router = $router;
        $this->slugger = $slugger;
    }

    // -------------------------------------------------------------------------

    /**
     * Zwraca gotowe metadatane dla strony głównej.
     *
     * @return string
     */
    public function getMeta()
    {
        $arr = [
            $this->metaTitleRender(),
            $this->metaDescriptionRender(),
            $this->metaKeywordsRender(),
            $this->canonicalRender(),
            $this->ogRender(),
        ];

        return implode(PHP_EOL.'    ', $arr).PHP_EOL;
    }

    /**
     * Zwraca tablicę metadanych dla strony głównej.
     *
     * @return array
     */
    public function getMetaArrayValues()
    {
        return [
            'title' => $this->getTitleGenerate(),
            'description' => $this->getDescriptionGenerate(),
            'keywords' => $this->getKeywordsGenerate(),
            'canonical' => $this->getCanonicalGenerate(),
            'image' => $this->getImageGenerate(),
        ];
    }

    // -------------------------------------------------------------------------

    /**
     * Generuje tytuł strony głównej.
     *
     * @return string
     */
    public function getTitleGenerate()
    {
        return htmlentities($this->config->get('metadata.title'));
    }

    /**
     * Generuje opis strony głównej.
     *
     * @return string
     */
    public function getDescriptionGenerate()
    {
        return htmlentities($this->config->get('metadata.description'));
    }

    /**
     * Generuje słowa kluczowe dla strony głównej.
     *
     * @return string
     */
    public function getKeywordsGenerate()
    {
        return $this->config->get('metadata.keywords');
    }

    /**
     * Generuje adres kanoniczny dla strony głównej.
     *
     * @return string
     */
    public function getCanonicalGenerate()
    {
        return $this->router->generate('homepage', [], UrlGeneratorInterface::ABSOLUTE_URL);
    }

    /**
     * Generuje zdjęcie strony głównej.
     *
     * @return string
     */
    public function getImageGenerate()
    {
        return $this->config->get('metadata.openGraphDefaultImage');
    }

    // -------------------------------------------------------------------------

    /**
     * Renderuje tytuł strony.
     *
     * @return string
     */
    public function metaTitleRender()
    {
        return sprintf('<title>%s</title>', $this->getTitleGenerate());
    }

    /**
     * Renderuje opis strony.
     *
     * @return string
     */
    public function metaDescriptionRender()
    {
        return sprintf(
            '<meta name="description" content="%s">',
            $this->getDescriptionGenerate()
        );
    }

    /**
     * Renderuje słowa kluczowe strony.
     *
     * @return string
     */
    public function metaKeywordsRender()
    {
        if (boolval(intval($this->config->get('metadata.keywordsEnable')))) {
            return sprintf(
                '<meta name="keywords" content="%s">',
                $this->getKeywordsGenerate()
            );
        }

        return;
    }

    /**
     * Renderuje kanoniczny adres URL.
     *
     * @return string
     */
    public function canonicalRender()
    {
        if (boolval(intval($this->config->get('metadata.canonical')))) {
            return sprintf(
                '<link rel="canonical" href="%s">',
                $this->getCanonicalGenerate()
            );
        }

        return;
    }

    // -------------------------------------------------------------------------

    /**
     * Renderuje OpenGraph.
     *
     * @return string
     */
    public function ogRender()
    {
        if (boolval(intval($this->config->get('metadata.openGraphEnable')))) {
            $arr = [
                $this->ogSiteNameRender(),
                $this->ogTitleRender(),
                $this->ogDescriptionRender(),
                $this->ogTypeRender(),
                $this->ogUrlRender(),
                $this->ogImageRender(),
            ];

            return trim(implode(PHP_EOL.'    ', $arr).PHP_EOL);
        }

        return;
    }

    /**
     * Renderuje nazwę strony dla OpenGraph.
     *
     * @return string
     */
    public function ogSiteNameRender()
    {
        return sprintf(
            '<meta property="og:site_name" content="%s">',
            $this->config->get('metadata.title')
        );
    }

    /**
     * Renderuje tytuł dla OpenGraph.
     *
     * @return string
     */
    public function ogTitleRender()
    {
        return sprintf(
            '<meta property="og:title" content="%s">',
            $this->config->get('metadata.title')
        );
    }

    /**
     * Renderuje opis strony dla OpenGraph.
     *
     * @return string
     */
    public function ogDescriptionRender()
    {
        return sprintf(
            '<meta property="og:description" content="%s">',
            $this->config->get('metadata.description')
        );
    }

    /**
     * Renderuje typ treści dla OpenGraph.
     *
     * @return string
     */
    public function ogTypeRender()
    {
        return sprintf(
            '<meta property="og:type" content="%s">',
            'website'
        );
    }

    /**
     * Renderuje adres URL dla OpenGraph.
     *
     * @return string
     */
    public function ogUrlRender()
    {
        return sprintf(
            '<meta property="og:url" content="%s">',
            $this->router->generate('homepage', [], UrlGeneratorInterface::ABSOLUTE_URL)
        );
    }

    /**
     * Renderuje zdjęcie dla OpenGraph.
     *
     * @return string
     */
    public function ogImageRender()
    {
        if (! empty($img = $this->getImageGenerate())) {
            return sprintf(
                '<meta property="og:image" content="%s">',
                $img
            );
        }

        return;
    }
}
