<?php

namespace Wi\Admin\CoreBundle\Service;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Wi\Admin\CoreBundle\Cache\CacheConfigManager;
use Wi\Admin\CoreBundle\Entity\Config as ConfigEntity;
use Wi\Admin\NewsBundle\Comment\CommentEngineInterface;

/**
 * Config.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class Config
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * @var CacheConfigManager
     */
    private $cacheManager;

    /**
     * @var array
     */
    private $config;

    /**
     * @var ContainerInterface
     */
    private $container;

    /**
     * Constructor.
     *
     * @param ContainerInterface $container
     * @param EntityManagerInterface $em
     * @param CacheConfigManager $cacheManager
     */
    public function __construct(
        ContainerInterface $container,
        EntityManagerInterface $em,
        CacheConfigManager $cacheManager
    ) {
        $this->container = $container;
        $this->em = $em;
        $this->cacheManager = $cacheManager;

        // Initialize configuration.
        $this->initConfig();
    }

    /**
     * Get config.
     *
     * @param string $key
     * @return string
     * @throws \Exception If a placeholder references a configuration that does not exist.
     */
    public function get($key)
    {
        if (array_key_exists($key, $this->config)) {
            return $this->config[$key];
        }

        throw new \Exception(sprintf('You have requested a non-existent configuration "%s".', $key));
    }

    /**
     * Get configuration.
     *
     * @return array
     */
    public function getConfig()
    {
        return $this->config;
    }

    /**
     * Zapis konfiguracji.
     *
     * @param string $groupName
     * @param string $name
     * @param string $value
     * @return void
     */
    public function save($groupName, $name, $value)
    {
        $config = $this->getRepository()->findOneBy([
            'groupName' => $groupName,
            'name' => $name,
        ]);

        if (! is_null($config) && $config->getValue() != $value) {
            $this->em->remove($config);

            $config = new ConfigEntity();
            $config
                ->setGroupName($groupName)
                ->setName($name)
                ->setValue($value)
            ;

            $this->em->persist($config);
            $this->em->flush($config);
            $this->cacheManager->clear();
        }
    }

    /**
     * Gets a parameter.
     *
     * @param string $name The parameter name
     * @return mixed The parameter value
     */
    public function getParameter($name)
    {
        return $this->container->getParameter($name);
    }

    /**
     * Czy komentarze WiCMS są włączone?
     *
     * @return bool
     */
    public function isWicmsCommentsEnabled()
    {
        return CommentEngineInterface::WICMS == $this->get('comment.enable');
    }

    /**
     * Czy komentarze DISQUS są włączone?
     *
     * @return bool
     */
    public function isDisqusCommentsEnabled() : bool
    {
        return CommentEngineInterface::DISQUS == $this->get('comment.enable');
    }

    /**
     * Czy komentarze FACEBOOK są włączone?
     *
     * @return bool
     */
    public function isFacebookCommentsEnabled() : bool
    {
        return CommentEngineInterface::FACEBOOK == $this->get('comment.enable');
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    /**
     * Initialize configuraction from database.
     */
    private function initConfig()
    {
        if (! $this->cacheManager->isHit()) { // Sprawdzenie, czy nie ma cache.
            // Sprawdzenie, czy istnieje tabela z ustawieniami.
            if (
                $this->em
                    ->getConnection()
                    ->getSchemaManager()
                    ->tablesExist($this->em
                        ->getClassMetadata('WiAdminCoreBundle:Config')
                        ->getTableName()
                    )
            ) {
                // Pobranie ustawień.
                $configs = $this->getRepository()->findAll();

                // Ustawienie parametrów.
                foreach ($configs as $config) {
                    $this->setConfig(
                        $config->getGroupName() . '.' . $config->getName(),
                        $config->getValue()
                    );
                }

                // Zapisanie cache.
                $this->cacheManager->set($this->config);
            }
        } else { // Jeśli jest cache.
            $this->config = $this->cacheManager->get();
        }
    }

    /**
     * Set config.
     *
     * @param string $key
     * @param string $value
     */
    private function setConfig($key, $value)
    {
        $this->config[$key] = $value;
    }

    /**
     * Pobiera repozytorium dla encji konfiguracyjnej.
     *
     * @return ConfigRepository
     */
    private function getRepository()
    {
        return $this->em->getRepository('WiAdminCoreBundle:Config');
    }
}
