<?php

namespace Wi\Admin\CoreBundle\Service;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Wi\Admin\CoreBundle\Service\Config;

/**
 * Cache.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class Cache
{
    private $config;
    private $htaccessPath;

    public function __construct(Config $config)
    {
        $this->config = $config;
        $this->htaccessPath = $this->config->getParameter('web_dir').'.htaccess';
    }


    public function rebuild()
    {
        $this->updateHtaccess();
    }

    // -------------------------------------------------------------------------

    private function updateHtaccess()
    {
        $htaccess = $this->htaccess();

        if (boolval(intval($this->config->get('cache.imageEnable')))) {
            $htaccess .= $this->images();
        }

        if (boolval(intval($this->config->get('cache.scriptEnable')))) {
            $htaccess .= $this->scripts();
        }

        if (boolval(intval($this->config->get('cache.styleEnable')))) {
            $htaccess .= $this->styles();
        }

        if (boolval(intval($this->config->get('cache.docEnable')))) {
            $htaccess .= $this->docs();
        }

        file_put_contents($this->htaccessPath, $htaccess);
    }

    private function docs()
    {
        $lifetime = $this->config->get('cache.docLifetime');

        $text = <<<DOC_HTACCESS
<FilesMatch ".(pdf)$">
    Header set Cache-Control "max-age=$lifetime, public, s-maxage=$lifetime"
</FilesMatch>

DOC_HTACCESS;

        return $text;
    }

    private function images()
    {
        $lifetime = $this->config->get('cache.imageLifetime');

        $text = <<<IMAGE_HTACCESS
<FilesMatch ".(jpg|jpeg|png|gif|tiff|ico)$">
    Header set Cache-Control "max-age=$lifetime, public, s-maxage=$lifetime"
</FilesMatch>

IMAGE_HTACCESS;

        return $text;
    }

    private function styles()
    {
        $lifetime = $this->config->get('cache.styleLifetime');

        $text = <<<STYLE_HTACCESS
<FilesMatch ".(css)$">
    Header set Cache-Control "max-age=$lifetime, public, s-maxage=$lifetime"
</FilesMatch>

STYLE_HTACCESS;

        return $text;
    }

    private function scripts()
    {
        $lifetime = $this->config->get('cache.scriptLifetime');

        $text = <<<SCRIPT_HTACCESS
<FilesMatch ".(js)$">
    Header set Cache-Control "max-age=$lifetime, public, s-maxage=$lifetime"
</FilesMatch>

SCRIPT_HTACCESS;

        return $text;
    }

    private function htaccess()
    {
        return <<<HTACCESS
DirectoryIndex app.php

<IfModule mod_negotiation.c>
    Options -MultiViews
</IfModule>

<IfModule mod_rewrite.c>
    RewriteEngine On

    RewriteCond %{REQUEST_URI}::$1 ^(/.+)/(.*)::\\2$
    RewriteRule ^(.*) - [E=BASE:%1]

    RewriteCond %{HTTP:Authorization} .
    RewriteRule ^ - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]

    RewriteCond %{ENV:REDIRECT_STATUS} ^$
    RewriteRule ^app\.php(?:/(.*)|$) %{ENV:BASE}/$1 [R=301,L]

    RewriteCond %{REQUEST_FILENAME} -f
    RewriteRule ^ - [L]

    RewriteRule ^ %{ENV:BASE}/app.php [L]
</IfModule>

<IfModule !mod_rewrite.c>
    <IfModule mod_alias.c>
        RedirectMatch 302 ^/$ /app.php/
    </IfModule>
</IfModule>


HTACCESS;
    }
}
