<?php

namespace Wi\Admin\CoreBundle\Service\Backup;

use Doctrine\ORM\EntityManagerInterface;
use Ifsnop\Mysqldump\Mysqldump;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Finder\Finder;
use Symfony\Component\HttpFoundation\File\File;

class BackupManager
{
    /**
     * @var string
     */
    private $backup_dir;

    private $format = 'YmdHis';
    private $file_ext = '.sql.gz';

    /**
     * @var mixed[]
     */
    private $params = [
        'host' => null,
        'port' => null,
        'dbname' => null,
        'user' => null,
        'password' => null,
        'uri' => 'mysql',
    ];

    /**
     * @var mixed[]
     */
    private $dumpSetting = [
        'compress' => Mysqldump::GZIP,
        'exclude-tables' => ['sessions'],
    ];

    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * Constructor.
     *
     * @param ContainerInterface $container
     */
    public function __construct(ContainerInterface $container, EntityManagerInterface $em)
    {
        $this->em = $em;
        $this->backup_dir = $container->getParameter('backup_dir');
        $this->params = array_merge($this->params, [
            'host' => $container->getParameter('database_host'),
            'port' => $container->getParameter('database_port'),
            'dbname' => $container->getParameter('database_name'),
            'user' => $container->getParameter('database_user'),
            'password' => $container->getParameter('database_password'),
        ]);
    }

    /**
     * Wykonuje kopię zapasową bazy danych.
     *
     * @return bool
     */
    public function backup()
    {
        $mysqldump = $this->initMysqldump();

        if (! is_dir($dir = $this->backup_dir)) {
            mkdir($dir);
        }

        $filepath = $dir.base64_encode('backup_'.date($this->format)).$this->file_ext;

        try {
            $mysqldump->start($filepath);
        } catch (\Exception $e) {
            return false;
        }

        return true;
    }

    /**
     * Zainicjowanie biblioteki Mysqldump.
     *
     * @return Mysqldump
     */
    public function initMysqldump()
    {
        return new Mysqldump($this->dsn(), $this->user(), $this->password(), $this->dumpSetting());
    }

    /**
     * Lista kopii bazy danych.
     *
     * @return array
     */
    public function getBackupList()
    {
        if (! is_dir($dir = $this->backup_dir)) {
            mkdir($dir);
        }

        $finder = new Finder();
        $finder
            ->in($this->backup_dir)
            ->files()
            ->name('*'.$this->file_ext)
        ;
        $out = [];

        foreach ($finder as $file) {
            $out[] = [
                'filename' => base64_decode(str_replace($this->file_ext, '', $file->getFilename())).$this->file_ext,
                'file' => $file,
            ];
        }

        return $out;
    }

    /**
     * Pobiera plik kopii na podstawie nazwy.
     *
     * @param string $filename
     * @return bool|File
     */
    public function getBackupFile($filename)
    {
        $filename = base64_encode(str_replace($this->file_ext, '', $filename)).$this->file_ext;

        if (! file_exists($filepath = $this->backup_dir.$filename)) {
            return false;
        }

        return new File($filepath);
    }
    /**
     * Usuwa plik kopii zapasowej.
     *
     * @param string $filename
     * @return bool
     */
    public function removeFile($filename)
    {
        $filename = base64_encode(str_replace($this->file_ext, '', $filename)).$this->file_ext;

        if (! file_exists($filepath = $this->backup_dir.$filename)) {
            return false;
        }

        return unlink($filepath);
    }

    /**
     * Przywraca kopię zapasową z pliku.
     *
     * @param string $filename
     * @return bool
     */
    public function restoreFile($filename)
    {
        $filename = base64_encode(str_replace($this->file_ext, '', $filename)).$this->file_ext;

        if (! file_exists($filepath = $this->backup_dir.$filename)) {
            return false;
        }

        $sql = implode('', gzfile($filepath));

        $metadata = $this->em->getMetadataFactory()->getAllMetadata();
        $tool = new \Doctrine\ORM\Tools\SchemaTool($this->em);

        $dropSchemaSql = $tool->getDropDatabaseSQL();
        unset($dropSchemaSql[array_search('DROP TABLE sessions', $dropSchemaSql)]);
        $stmt = $this->em->getConnection();

        foreach ($dropSchemaSql as $dropSql) {
            $stmt = $this->em->getConnection()->prepare($dropSql);
            $stmt->execute();
        }

        $stmt = $this->em->getConnection()->prepare($sql);
        $stmt->execute();

        return true;
    }

    // ------------------------------------------------------------
    // Private functions.
    // ------------------------------------------------------------

    private function dsn()
    {
        return $this->params['uri'].':host='.$this->params['host'].';dbname='.$this->params['dbname'];
    }

    private function user()
    {
        return $this->params['user'];
    }

    private function password()
    {
        return $this->params['password'];
    }

    private function dumpSetting()
    {
        return $this->dumpSetting;
    }
}
