<?php

namespace Wi\Admin\ContactBundle\Service;

use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Wi\Admin\ContactBundle\Entity\Form;
use Wi\Admin\ContactBundle\Entity\Form\FormFieldTypeInterface;
use Wi\Admin\CoreBundle\Service\Config;

class FormGenerator
{
    const RECAPTCHA_API_JS = 'https://www.google.com/recaptcha/api.js';
    const RECAPTCHA_SITE_VERIFY = 'https://www.google.com/recaptcha/api/siteverify';

    /**
     * @var Config
     */
    private $config;

    /**
     * @var RouterInterface
     */
    private $router;

    /**
     * @var string
     */
    private $recaptchaSiteKey;

    /**
     * @var string
     */
    private $recaptchaSecretKey;

    /**
     * Constructor.
     *
     * @param Config $config
     * @param RouterInterface $router
     */
    public function __construct(Config $config, RouterInterface $router)
    {
        $this->config = $config;
        $this->router = $router;

        $this->recaptchaSiteKey = $this->config->get('recaptcha.siteKey');
        $this->recaptchaSecretKey = $this->config->get('recaptcha.secretKey');
    }

    /**
     * Generowanie formularza.
     *
     * @param Form|null $form
     * @return string|null
     */
    public function generate(Form $form = null)
    {
        if (is_null($form)) {
            return null;
        }

        $renderedForm = '
        <div class="contact_form">
            <form action="'.$this->router->generate('wi_front_contact_receiver_form', [], UrlGeneratorInterface::ABSOLUTE_URL).'" method="POST" accept-charset="utf-8" id="form_'.$form->getId().'">
                <input type="hidden" name="email_repeat_123_">
                <input type="hidden" name="form_id" value="'.$form->getId().'">
        ';

        foreach ($form->getFields() as $field) {
            $renderedForm .=  $this->renderField($field);
        }

        $renderedForm .= $this->getRecaptcha($form);
        $renderedForm .= '
                <input type="submit" class="contact_form_submit button" value="Wyślij">
            </form>
        </div>
        ';

        return $renderedForm;
    }

    /**
     * Przygotowanie recaptchy, jeśli formularz z niej korzysta.
     *
     * @param Form $form
     * @return string|null
     */
    public function getRecaptcha(Form $form)
    {
        if ($form->getRecaptcha()) {
            if (! empty($this->recaptchaSiteKey) && ! empty($this->recaptchaSecretKey)) {
                $renderRecaptcha = '
                    <div class="recaptcha-wrapper">
                        <script src="'.self::RECAPTCHA_API_JS.'" async defer></script>
                        <div class="g-recaptcha" data-sitekey="'.$this->recaptchaSiteKey.'"></div>
                    </div>
                ';

                return $renderRecaptcha;
            }

            return null;
        }

        return null;
    }

    /**
     * Renderowanie wiersza pola formularza.
     *
     * @param FormField $field
     */
    public function renderField($field)
    {
        $id = 'contact_form_field_'.$field->getId();
        $format = $this->getBlockFieldFormat();

        return sprintf($format, $this->getLabel($field, $id), $this->getField($field, $id));
    }

    /**
     * Generowanie etykiety pola formularza.
     *
     * @param FormField $field
     * @param string $id
     * @return string
     */
    public function getLabel($field, $id)
    {
        return '<label for="'.$id.'">'.$field->getLabel().'</label>';
    }

    /**
     * Generowanie pola formularza.
     *
     * @param FormField $field
     * @param string $id
     * @return string
     */
    public function getField($field, $id)
    {
        $value = ! is_null($field->getValue()) ? ' value="'.$field->getValue().'"' : null;
        $placeholder = ! is_null($field->getPlaceholder()) ? ' placeholder="'.$field->getPlaceholder().'"' : null;
        $required = $field->getRequired() ? ' required="required"' : null;
        $name = $field->getName();

        $fieldType = $field->getFormFieldType();
        $class = ! is_null($fieldType->getClass()) ? ' class="'.$fieldType->getClass().'"' : null;
        $pattern = ! is_null($fieldType->getPattern()) ? ' pattern="'.$fieldType->getPattern().'"' : null;
        $patternTitle = ! is_null($fieldType->getPatternTitle()) ? ' title="'.$fieldType->getPatternTitle().'"' : null;

        switch ($fieldType->getType()) {
            case FormFieldTypeInterface::INPUT_CHECKBOX:
                $format = '<input type="checkbox" name="%s"%s%s%s%s%s%s%s>';
                return sprintf($format, $name, $class, $pattern, $patternTitle, ' id="'.$id.'"', $placeholder, $required, $value);

            case FormFieldTypeInterface::INPUT_EMAIL:
                $format = '<input type="email" name="%s"%s%s%s%s%s%s%s>';
                return sprintf($format, $name, $class, $pattern, $patternTitle, ' id="'.$id.'"', $placeholder, $required, $value);

            case FormFieldTypeInterface::INPUT_TEL:
                $format = '<input type="tel" name="%s"%s%s%s%s%s%s%s>';
                return sprintf($format, $name, $class, $pattern, $patternTitle, ' id="'.$id.'"', $placeholder, $required, $value);

            case FormFieldTypeInterface::INPUT_TEXT:
                $format = '<input type="text" name="%s"%s%s%s%s%s%s%s>';
                return sprintf($format, $name, $class, $pattern, $patternTitle, ' id="'.$id.'"', $placeholder, $required, $value);

            case FormFieldTypeInterface::TEXTAREA:
                $format = '<textarea name="%s"%s%s%s%s%s%s cols="50" rows="10">%s</textarea>';
                return sprintf($format, $name, $class, $pattern, $patternTitle, ' id="'.$id.'"', $placeholder, $required, $field->getValue());

            default:
                return null;
        }
    }

    /**
     * Format bloku pola formularza.
     *
     * @return string
     */
    public function getBlockFieldFormat()
    {
        return '<div class="contact_form_row">%s%s</div>';
    }

    public function verifyRecaptcha($response)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, self::RECAPTCHA_SITE_VERIFY);
        curl_setopt($ch, CURLOPT_HEADER, 0);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, [
            'secret' => $this->recaptchaSecretKey,
            'response' => $response,
        ]);
        $response = curl_exec($ch);
        curl_close($ch);
        $data = json_decode($response);

        return $data->success;
    }
}
