<?php

namespace Wi\Admin\ContactBundle\Form;

use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\Form\Extension\Core\Type\CollectionType;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvent;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\OptionsResolver;
use Wi\Admin\ContactBundle\Entity\Form;
use Wi\Admin\ContactBundle\Form\FormFieldType;
use Wi\Admin\PageBundle\Form\PageType;

/**
 * Form edit type.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class FormEditType extends AbstractType
{
    /**
     * @var EntityManagerInterface
     */
    private $em;

    /**
     * Construct.
     *
     * @param EntityManagerInterface $em
     */
    public function __construct(EntityManagerInterface $em)
    {
        $this->em = $em;
    }

    /**
     * {@inheritdoc}
     */
    public function buildForm(FormBuilderInterface $builder, array $options)
    {
        $form = $builder->getData();

        $builder
            ->add('name', TextType::class, [
                'attr' => [
                    'placeholder' => 'contactNew.name',
                ],
                'label' => 'contactNew.nameLabel',
            ])
            ->add('type', ChoiceType::class, [
                'disabled' => true,
                'choices' => [
                    'contactNew.choiseContact' => 'C',
                    'contactNew.choiseCTA' => 'L',
                ],
                'label' => 'contactNew.typeLabel',
            ])
            ->add('recipients', TextType::class, [
                'attr' => [
                    'placeholder' => 'contactEdit.recipientsLabel',
                ],
                'label' => 'contactEdit.recipientsLabel',
                'required' => false
            ])
            ->add('autoreply', null, [
                'attr' => [
                    'class' => 'switch-input replayInput'
                ],
                'label' => 'contactEdit.replayLabel',
            ])
            ->add('recaptcha', null, [
                'attr' => [
                    'class' => 'switch-input recaptchaInput'
                ],
                'label' => 'contactEdit.recaptchaLabel',
            ])
            ->add('replyMessage', TextareaType::class, [
                'attr' => [
                    'class' => 'tinyMCEMail',
                ],
                'required' => false,
                'label' => 'contactEdit.replyMessage',
            ])
            ->add('replySubject', TextType::class, [
                'attr' => [
                    'placeholder' => 'contactEdit.replySubject',
                ],
                'label' => 'contactEdit.replySubject',
                'required' => false
            ])
            ->add('fields', CollectionType::class, [
                'allow_add' => true,
                'allow_delete' => true,
                'by_reference' => false,
                'entry_options' => [],
                'entry_type' => FormFieldType::class,
                'label' => false,
            ])
            ->add('siteKey', null, [
                'attr' => [
                    'placeholder' => 'contactEdit.siteKey',
                ],
                'data' => $this->getConfig('recaptcha', 'siteKey'),
                'label' => 'contactEdit.siteKey',
                'mapped' => false,
            ])
            ->add('secretKey', null, [
                'attr' => [
                    'placeholder' => 'contactEdit.secretKey',
                ],
                'data' => $this->getConfig('recaptcha', 'secretKey'),
                'label' => 'contactEdit.secretKey',
                'mapped' => false,
            ])
            ->add('save', SubmitType::class,[
                'attr' => [
                    'class' => 'button hide tiny saveButtonPage',
                ],
                'label' => 'contactEdit.savePage',
            ])
        ;

        // Jeśli formularz jest formularzem kontaktowym.
        if ($form->getType() == FORM::TYPE_CONTACT) {
            $builder
                ->add('page', EntityType::class, [
                    'attr' => [
                        'class' => 'select2page',
                    ],
                    'choice_label' => 'title',
                    'class' => 'WiAdminPageBundle:Page',
                    'label' => 'contactEdit.chosePage',
                    'placeholder' => 'contactEdit.choosePage',
                    'query_builder' => function (EntityRepository $er) {
                        return $er->createQueryBuilder('p')
                            ->orderBy('p.title', 'ASC')
                        ;
                    },
                    'required' => false,
                    'disabled' => false,
                ])
                ->add('maps', CollectionType::class, [
                    'allow_add' => true,
                    'allow_delete' => true,
                    'by_reference' => false,
                    'entry_options' => [],
                    'entry_type' => FormMapType::class,
                    'label' => false,
                ])
            ;

            if (! is_null($form->getPage())) {
                $builder
                    ->add('pageTitle', TextType::class, [
                        'property_path' => 'page.title',
                        'attr' => [
                            'maxlength' => 256,
                            'placeholder' => 'contactEdit.titleTitle',
                        ],
                        'label' => 'contactEdit.titleTitle',
                    ])
                    ->add('pageContent', TextareaType::class, [
                        'property_path' => 'page.content',
                        'attr' => [
                            'maxlength' => 65535,
                            'rows' => 10,
                            'class' => 'tinyMCE',
                        ],
                        'label' => 'contactEdit.contentPage',
                        'required' => false,
                    ])
                    ->addEventListener(FormEvents::PRE_SUBMIT, function (FormEvent $event) {
                        $data = $event->getData();
                        $viewData = $event->getForm()->getViewData();
                        $form = $event->getForm();

                        if (! $data) {
                            return;
                        }

                        if (empty($data['page'])) {
                            $form->remove('pageTitle');
                            $form->remove('pageContent');
                            unset($data['pageTitle']);
                            unset($data['pageContent']);
                            $event->setData($data);
                        } else {
                            if (isset($data['page']) && ! is_null($viewData->getPage()) && intval($data['page']) != $viewData->getPage()->getId()) {
                                $page = $this->getPageRepository()->find(intval($data['page']));
                                $data['pageTitle'] = $page->getTitle();
                                $data['pageContent'] = $page->getContent();
                                $event->setData($data);
                            }
                        }
                    })
                ;
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function configureOptions(OptionsResolver $resolver)
    {
        $resolver->setDefaults([
            'data_class' => Form::class,
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function getBlockPrefix()
    {
        return 'wi_admin_contactbundle_form';
    }

    /**
     * Pobiera repozytorium stron.
     *
     * @return PageRepository
     */
    private function getPageRepository()
    {
        return $this->em->getRepository('WiAdminPageBundle:Page');
    }

    /**
     * Pobiera wartość parametru z konfiguracji.
     *
     * @param string $groupName
     * @param string $name
     * @return string
     */
    private function getConfig($groupName, $name)
    {
        $config = $this->em->getRepository('WiAdminCoreBundle:Config')
            ->findOneBy([
                'groupName' => $groupName,
                'name' => $name,
            ])
        ;

        if (is_null($config)) {
            return $config;
        }

        return $config->getValue();
    }
}
