<?php

namespace Wi\Admin\ContactBundle\Entity;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Symfony\Component\Validator\Constraints as Assert;
use Wi\Admin\ContactBundle\Entity\FormField;
use Wi\Admin\ContactBundle\Entity\FormMap;
use Wi\Admin\PageBundle\Entity\Page;

/**
 * Form
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 *
 * @ORM\Table(name="contact_forms")
 * @ORM\HasLifecycleCallbacks()
 * @ORM\Entity(repositoryClass="Wi\Admin\ContactBundle\Repository\FormRepository")
 * @Gedmo\SoftDeleteable(fieldName = "dateDeleted")
 */
class Form
{
    /**
     * Oznaczenie typu formularza, jako formularz kontaktowy.
     *
     * @const TYPE_CONTACT
     */
    const TYPE_CONTACT = 'C';

    /**
     * Oznaczenie typu formularza, jako formularz lead'ów.
     *
     * @const TYPE_LEAD
     */
    const TYPE_LEAD = 'L';

    /**
     * @var int
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var int
     *
     * @ORM\ManyToOne(targetEntity="Wi\Admin\PageBundle\Entity\Page", fetch="EAGER")
     * @ORM\JoinColumn(name="page_id", referencedColumnName="id", nullable=true)
     */
    private $page;

    /**
     * @var string
     *
     * Dostepne typy:
     *  C - formularz kontaktowy
     *  L - formularz lead'ów
     *
     * @ORM\Column(name="type", type="string", length=1)
     * @Assert\NotBlank()
     */
    private $type;

    /**
     * @var string
     *
     * @ORM\Column(name="name", type="string", length=45)
     * @Assert\NotBlank()
     */
    private $name;

    /**
     * @var string
     *
     * @ORM\Column(name="recipients", type="string", length=256, nullable=true)
     */
    private $recipients;

    /**
     * @var bool
     *
     * @ORM\Column(name="autoreply", type="boolean")
     */
    private $autoreply;

    /**
     * @var string
     *
     * @ORM\Column(name="replySubject", type="string", length=256, nullable=true)
     */
    private $replySubject;

    /**
     * @var string
     *
     * @ORM\Column(name="replyMessage", type="text", nullable=true)
     */
    private $replyMessage;

    /**
     * @var bool
     *
     * @ORM\Column(name="recaptcha", type="boolean")
     */
    private $recaptcha;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateCreated", type="datetime")
     */
    private $dateCreated;

    /**
     * @var \DateTime
     *
     * @ORM\Column(name="dateDeleted", type="datetime", nullable=true)
     */
    private $dateDeleted;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\ContactBundle\Entity\FormField", mappedBy="form", cascade={"persist"})
     * @ORM\OrderBy({"orderNr" = "ASC"})
     */
    private $fields;

    /**
     * @var ArrayCollection
     *
     * @ORM\OneToMany(targetEntity="Wi\Admin\ContactBundle\Entity\FormMap", mappedBy="form", cascade={"persist"})
     * @ORM\OrderBy({"orderNr" = "ASC"})
     */
    private $maps;


    // ------------------------------------------------------------
    // Has Lifecycle Callbacks methods.
    // ------------------------------------------------------------

    /**
     * Funkcja wykonuje się podczas wstawiania nowego rekordu do bazy.
     *
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        // Ustawienie daty utworzenia.
        $this->dateCreated = new \DateTime();

        // Ustawienie autoodpowiedzi.
        if (is_null($this->autoreply)) {
            $this->autoreply = false;
        }

        // Ustawienie recaptchy.
        if (is_null($this->recaptcha)) {
            $this->recaptcha = false;
        }
    }

    // ------------------------------------------------------------
    // Getters and setters methods.
    // ------------------------------------------------------------

    /**
     * Set id
     *
     * @param int $id
     * @return Form
     */
    public function setId($id)
    {
        $this->id = $id;

        return $this;
    }

    /**
     * Get id
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set page
     *
     * @param Page $page
     * @return Form
     */
    public function setPage(Page $page = null)
    {
        $this->page = $page;

        return $this;
    }

    /**
     * Get page
     *
     * @return Page
     */
    public function getPage()
    {
        return $this->page;
    }

    /**
     * Set type
     *
     * @param string $type
     * @return Form
     */
    public function setType($type)
    {
        $this->type = $type;

        return $this;
    }

    /**
     * Get type
     *
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * Set name
     *
     * @param string $name
     * @return Form
     */
    public function setName($name)
    {
        $this->name = $name;

        return $this;
    }

    /**
     * Get name
     *
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Set recipients
     *
     * @param string $recipients
     * @return Form
     */
    public function setRecipients($recipients)
    {
        $this->recipients = $recipients;

        return $this;
    }

    /**
     * Get recipients
     *
     * @return string
     */
    public function getRecipients()
    {
        return $this->recipients;
    }

    /**
     * Set autoreply
     *
     * @param boolean $autoreply
     * @return Form
     */
    public function setAutoreply($autoreply)
    {
        $this->autoreply = $autoreply;

        return $this;
    }

    /**
     * Get autoreply
     *
     * @return bool
     */
    public function getAutoreply()
    {
        return $this->autoreply;
    }

    /**
     * Set replySubject
     *
     * @param string $replySubject
     * @return Form
     */
    public function setReplySubject($replySubject)
    {
        $this->replySubject = $replySubject;

        return $this;
    }

    /**
     * Get replySubject
     *
     * @return string
     */
    public function getReplySubject()
    {
        return $this->replySubject;
    }

    /**
     * Set replyMessage
     *
     * @param string $replyMessage
     * @return Form
     */
    public function setReplyMessage($replyMessage)
    {
        $this->replyMessage = $replyMessage;

        return $this;
    }

    /**
     * Get replyMessage
     *
     * @return string
     */
    public function getReplyMessage()
    {
        return $this->replyMessage;
    }

    /**
     * Set recaptcha
     *
     * @param boolean $recaptcha
     * @return Form
     */
    public function setRecaptcha($recaptcha)
    {
        $this->recaptcha = $recaptcha;

        return $this;
    }

    /**
     * Get recaptcha
     *
     * @return bool
     */
    public function getRecaptcha()
    {
        return $this->recaptcha;
    }

    /**
     * Set dateCreated
     *
     * @param \DateTime $dateCreated
     * @return Form
     */
    public function setDateCreated($dateCreated)
    {
        $this->dateCreated = $dateCreated;

        return $this;
    }

    /**
     * Get dateCreated
     *
     * @return \DateTime
     */
    public function getDateCreated()
    {
        return $this->dateCreated;
    }

    /**
     * Set dateDeleted
     *
     * @param \DateTime $dateDeleted
     * @return Form
     */
    public function setDateDeleted($dateDeleted)
    {
        $this->dateDeleted = $dateDeleted;

        return $this;
    }

    /**
     * Get dateDeleted
     *
     * @return \DateTime
     */
    public function getDateDeleted()
    {
        return $this->dateDeleted;
    }

    /**
     * Add field
     *
     * @param FormField $field
     * @return Form
     */
    public function addField(FormField $field)
    {
        $field->setForm($this);
        $this->fields[] = $field;

        return $this;
    }

    /**
     * Remove field
     *
     * @param FormField $field
     */
    public function removeField(FormField $field)
    {
        $this->fields->removeElement($field);
    }

    /**
     * Get fields
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getFields()
    {
        return $this->fields;
    }

    /**
     * Add map
     *
     * @param FormMap $map
     * @return Form
     */
    public function addMap(FormMap $map)
    {
        $map->setForm($this);
        $this->maps[] = $map;

        return $this;
    }

    /**
     * Remove map
     *
     * @param FormMap $map
     */
    public function removeMap(FormMap $map)
    {
        $this->maps->removeElement($map);
    }

    /**
     * Get maps
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getMaps()
    {
        return $this->maps;
    }

    // ------------------------------------------------------------
    // Methods.
    // ------------------------------------------------------------

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->fields = new ArrayCollection();
        $this->maps = new ArrayCollection();
    }

    /**
     * Pobiera nazwę route'a w zależności od typu formularza.
     *
     * @return string
     */
    public function getRouteName()
    {
        switch ($this->type) {
            case 'C': return 'wi_admin_contact_messages';
            case 'L': return 'wi_admin_contact_leads';
        }
    }
}
