<?php

namespace Wi\Admin\AdvertisingBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\AdvertisingBundle\Entity\Advertisement;
use Wi\Admin\AdvertisingBundle\Entity\AdvertisingBlock;
use Wi\Admin\AdvertisingBundle\Form\AdvertisementType;
use Wi\Admin\AdvertisingBundle\Form\AdvertisingBlockType;
use Wi\Admin\AdvertisingBundle\Utils\UploadAdvertisement;
use Wi\Admin\CoreBundle\Form\DeleteType;

class AdvertisingBlockController extends Controller
{
    /**
     * Lista bloków reklamowych wraz z reklamami do nich przypisanymi.
     *
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function indexAction()
    {
        // Utworzenie instancji EntityManager-a.
        $em = $this->getDoctrine()->getManager();
        $advertisingBlocks = $em->getRepository('WiAdminAdvertisingBundle:AdvertisingBlock')->findAll();

        return $this->render('WiAdminAdvertisingBundle:AdvertisingBlock:index.html.php', [
            'advertisingBlocks' => $advertisingBlocks,
        ]);
    }

    /**
     * Tworzenie nowej reklamy.
     *
     * @param Request $request
     * @param UploadAdvertisement $uploader
     * @param AdvertisingBlock $advertisingBlock
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function newAction(Request $request, UploadAdvertisement $uploader, AdvertisingBlock $advertisingBlock)
    {
        $advertisement = new Advertisement();
        $advertisement->setAdvertisingBlock($advertisingBlock);

        $form = $this->createForm(AdvertisementType::class, $advertisement);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            if ($file = $form->get('file')->getData()) {
                $filename = $uploader->upload($file);
                $advertisement->setImage($filename);
            }

            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();
            $em->persist($advertisement);
            $em->flush();

            return $this->redirectToRoute('wi_admin_advertising_block_index');
        }

        return $this->render('WiAdminAdvertisingBundle:AdvertisingBlock:new.html.php', [
            'advertisingBlock' => $advertisingBlock,
            'form' => $form->createView(),
        ]);
    }

    /**
     * Edycja istniejącej reklamy.
     *
     * @param Request $request
     * @param Advertisement $advertisement
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function editAction(Request $request, Advertisement $advertisement)
    {
        $form = $this->createForm(AdvertisementType::class, $advertisement);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            if ($file = $form->get('file')->getData()) {
                $filename = $uploader->upload($file);
                $advertisement->setImage($filename);
            }

            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();
            $em->persist($advertisement);
            $em->flush();

            return $this->redirectToRoute('wi_admin_advertising_block_index');
        }

        return $this->render('WiAdminAdvertisingBundle:AdvertisingBlock:edit.html.php', [
            'advertisingBlock' => $advertisement->getAdvertisingBlock(),
            'form' => $form->createView(),
            'advertisement' => $advertisement
        ]);
    }

    /**
     * Usunięcie istniejącej reklamy.
     *
     * @param Request $request
     * @param Advertisement $advertisement
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function deleteAction(Request $request, Advertisement $advertisement)
    {
        $form = $this->createForm(DeleteType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();
            $em->remove($advertisement);
            $em->flush();

            return $this->redirectToRoute('wi_admin_advertising_block_index');
        }

        return $this->render('WiAdminAdvertisingBundle:AdvertisingBlock:delete.html.php', [
            'form' => $form->createView(),
            'advertisement' => $advertisement,
        ]);
    }

    /**
     * Tworzenie nowego bloku reklamowego.
     *
     * @param Request $request
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function blockNewAction(Request $request)
    {
        $advertisingBlock = new AdvertisingBlock();
        $form = $this->createForm(AdvertisingBlockType::class, $advertisingBlock);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();
            $em->persist($advertisingBlock);
            $em->flush();

            return $this->redirectToRoute('wi_admin_advertising_block_index');
        }

        return $this->render('WiAdminAdvertisingBundle:AdvertisingBlock/Block:new.html.php', [
            'form' => $form->createView(),
        ]);
    }

    /**
     * Edycja bloku reklamowego.
     *
     * @param Request $request
     * @param AdvertisingBlock $advertisingBlock
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function blockEditAction(Request $request, AdvertisingBlock $advertisingBlock)
    {
        $form = $this->createForm(AdvertisingBlockType::class, $advertisingBlock);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();
            $em->persist($advertisingBlock);
            $em->flush();

            return $this->redirectToRoute('wi_admin_advertising_block_index');
        }

        return $this->render('WiAdminAdvertisingBundle:AdvertisingBlock/Block:edit.html.php', [
            'form' => $form->createView(),
            'advertisingBlock' => $advertisingBlock,
        ]);
    }

    /**
     * Usunięcie bloku reklamowego.
     *
     * @param Request $request
     * @param AdvertisingBlock $advertisingBlock
     * @return \Symfony\Component\HttpFoundation\Response
     */
    public function blockDeleteAction(Request $request, AdvertisingBlock $advertisingBlock)
    {
        $form = $this->createForm(DeleteType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();
            $em->remove($advertisingBlock);
            $em->flush();

            return $this->redirectToRoute('wi_admin_advertising_block_index');
        }

        return $this->render('WiAdminAdvertisingBundle:AdvertisingBlock/Block:delete.html.php', [
            'form' => $form->createView(),
            'advertisingBlock' => $advertisingBlock,
        ]);
    }
}
