<?php

namespace Wi\Admin\AdminBundle\Controller;

use DeviceDetector\DeviceDetector;
use DeviceDetector\Parser\Device\DeviceParserAbstract;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\Form\FormError;
use Symfony\Component\HttpFoundation\Request;
use Wi\Admin\AdminBundle\Entity\Administrator;
use Wi\Admin\AdminBundle\Form\ForgotPasswordEmailType;
use Wi\Admin\AdminBundle\Form\ResetPasswordType;
use Wi\Admin\AdminBundle\Service\Mailer;

/**
 * Security controller.
 *
 * @author Jakub Nowak <jakub.nowak@webimpuls.pl>
 * @copyright 2017 WEBimpuls Sp. z o.o.
 */
class SecurityController extends Controller
{
    /**
     * Logowanie.
     *
     * @param   Request $request
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function loginAction(Request $request)
    {
        // Przekierowanie do panelu, jeśli administrator jest zalogowany.
        if ($this->get('security.authorization_checker')->isGranted('ROLE_ADMIN')) {
            return $this->redirectToRoute('wi_admin_admin_dashboard');
        }

        // Get authentication utils.
        $authenticationUtils = $this->get('security.authentication_utils');

        // Get the login error if there is one.
        $error = $authenticationUtils->getLastAuthenticationError();

        // Last username entered by the user.
        $lastUsername = $authenticationUtils->getLastUsername();

        // Rozpoznawanie urządzenia.
        $dd = new DeviceDetector($request->server->get('HTTP_USER_AGENT'));
        $dd->parse();
        $browser = $dd->getClient();

        return $this->render('WiAdminAdminBundle:Security:login.html.php', [
            'error'         => $error,
            'last_username' => $lastUsername,
            'browser'       => $browser,
        ]);
    }

    /**
     * Przywracanie hasła.
     *
     * @param   Request $request
     * @param   Mailer  $mailer
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function forgotPasswordAction(Request $request, Mailer $mailer)
    {
        // Utworzenie formularza.
        $forgotForm = $this->createForm(ForgotPasswordEmailType::class);
        $forgotForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($forgotForm->isSubmitted() && $forgotForm->isValid()) {
            $email = $forgotForm->get('email')->getData();

            // Utworzenie instancji EntityManager-a.
            $em = $this->getDoctrine()->getManager();

            // Pobranie administratora na podstawie adresu e-mail.
            $administrator = $em->getRepository('WiAdminAdminBundle:Administrator')
                ->findOneByEmail($email)
            ;

            // Sprawdzenie, czy istnieje administrator o podanym adresie e-mail.
            if (is_null($administrator)) {
                // Dodanie błędu, jeśli administrator nie istnieje.
                $forgotForm->addError(
                    new FormError(
                        $this->get('translator')
                            ->trans('forgot.admin_by_email_not_found')
                    )
                );
            } else {
                // Wygenerowania hasha.
                $hash = hash('sha256',
                    md5(
                        time()
                        . $this->get('session')->getId()
                        . $administrator->getEmail()
                        . time()
                        . rand()
                    )
                );

                // Ustawienie hasha dla administratora.
                $administrator->setHash($hash);

                // Ustawienie daty ważności hasha.
                $administrator->setHashExpired((new \DateTime())->modify('+1 day'));

                // Zapis.
                $em->persist($administrator);
                $em->flush();

                // Wysłanie wiadomości.
                $res = $mailer->sendForgotPasswordEmail($administrator);

                // Ustawienie komunikatu o powodzeniu lub błędzie.
                if ($res) {
                    $this->addFlash(
                        'forgot.success',
                        $this->get('translator')->trans('forgot.succes')
                    );
                } else {
                    $this->addFlash(
                        'forgot.error',
                        $this->get('translator')->trans('forgot.error')
                    );
                }
            }
        }

        return $this->render('WiAdminAdminBundle:Security:forgot_password.html.php', [
            'formForgot' => $forgotForm->createView(),
        ]);
    }

    /**
     * Resetowanie hasła.
     *
     * @param   Request         $request
     * @param   Administrator   $administrator  Default 'null'.
     * @return  \Symfony\Component\HttpFoundation\Response
     */
    public function resetPasswordAction(Request $request, Administrator $administrator = null)
    {
        if (is_null($administrator) || ($administrator->getHashExpired() < new \DateTime())) {
            throw $this->createNotFoundException('Podana strona nie została odnaleziona.');
        }

        // Utworzenie formularza.
        $resetPasswordForm = $this->createForm(ResetPasswordType::class, $administrator);
        $resetPasswordForm->handleRequest($request);

        // Weryfikacja, czy formularz został przesłany i pomyślnie zwalidowany.
        if ($resetPasswordForm->isSubmitted() && $resetPasswordForm->isValid()) {
            if (! empty($resetPasswordForm->get('password')->getData())) {
                // Ustawienie hasła administratora.
                $administrator->setPassword(
                    $this->get('security.password_encoder')
                        ->encodePassword(
                            $administrator,
                            $administrator->getPassword()
                        )
                );

                // Ustawienie pustego hasha dla administratora.
                $administrator->setHash(null);

                // Ustawienie pustej daty ważności hasha.
                $administrator->setHashExpired(null);

                // Utworzenie instancji EntityManager-a.
                $em = $this->getDoctrine()->getManager();

                // Zapis.
                $em->persist($administrator);
                $em->flush();

                // Ustawienie komunikatu o zmianie hasła.
                $this->addFlash('reset.success', 'Hasło zostało zmienione.');
            } else {
                // Dodanie błędu, jeśli hasło jest puste.
                $resetPasswordForm->addError(
                    new FormError(
                        $this->get('translator')
                            ->trans('forgot.password_must_not_empty')
                    )
                );
            }
        }

        return $this->render('WiAdminAdminBundle:Security:reset_password.html.php', [
            'formResetPassword' => $resetPasswordForm->createView(),
        ]);
    }

    /**
     * ----------------------------------------------------------------------
     * Private functions.
     * ----------------------------------------------------------------------
     */

    /**
     * Tworzenie treści wiadomości dla przypomnienia hasła.
     *
     * @param   array   $data
     * @return  \Swift_Message
     */
    private function createForgotPasswordMessage($data)
    {
        return \Swift_Message::newInstance()
            ->setSubject('Resetuj hasło')
            ->setFrom([
                // Adres e-mail => Nazwa.
                $this->getParameter('mailer_user') => $this->getParameter('mailer_user'),
            ])
            ->setTo($data['administrator']->getEmail())
            ->setBody(
                $this->renderView('WiAdminAdminBundle:templates:emails/forgot_password.html.php', [
                    'administrator' => $data['administrator'],
                ]),
                'text/html'
            )
        ;
    }
}
